import React, { useMemo, useRef, useState } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import useAsyncEffect from 'use-async-effect';
import { Checkbox, Divider, Form, Input, Radio, Typography } from 'antd';
import { CheckCircleOutlined, QuestionCircleOutlined } from '@ant-design/icons';
import { DOCUMENT_TYPES, HIDE_AUTOFILL_VERDICTS, VERDICT_OK, VERDICT_OTHER } from '../../domain/documents';
import { translate } from '../../utils/translate';
import { convertFormFieldsToValues } from '../../utils/converters';
import { Hint } from '../Utility/Hint';
import { withHotkey } from '../Utility/WithHotkey';
import { PassportBio } from './PassportBio';
import { PassportReg } from './PassportReg';
import { LicenseFront } from './LicenseFront';
import { LicenseBack } from './LicenseBack';
import { CountryPicker } from './CountryPicker';
import { Criteria } from './Criteria';
import { Spin } from '../Spin';
import './Document.less';

export const formLayout = {
    labelCol: { span: 5 },
    wrapperCol: { offset: 1, span: 18 },
};

export const DocumentView = ({
    documentType,
    fieldsData,
    isVerified,
    onFieldsChange,
    facts,
    scaleOptions,
    verdicts,
    onVerdictChange,
    taskError,
    resetTaskError,
    activeDocumentIndex,
    documentIndex,
    onAddressSearch,
    isForeign,
    submitting,
    formsData,
    isImgValid
}) => {
    const verdictsFormName = `verdicts_${documentType}`;
    const [verdictsForm] = Form.useForm();
    const [documentForm] = Form.useForm();
    const commentRef = useRef(null);

    const [isStrictValidation, setIsStrictValidation] = useState(false);

    const hotkeyEventsDisabled = documentIndex !== activeDocumentIndex || submitting;

    useAsyncEffect(async () => {
        try {
            await documentForm.validateFields();
        } catch {
            // just to shut up the browser warnings, don't care about this promise
        }
    }, [isStrictValidation]);

    const initialValues = useMemo(
        () =>
            Object.entries(fieldsData).reduce(
                (values, [fieldName, fieldValue]) => ({
                    ...values,
                    [fieldName]: fieldValue ?? '',
                }),
                {},
            ),
        [],
    );

    const availableFields = useMemo(() => Object.keys(formsData), []);

    const fields = Object.entries(facts).map(([name, label]) => ({
        name,
        label,
    }));

    if (isImgValid === false) {
        fields.unshift({name: 'LOADING_ERROR', label: 'Ошибка загрузки фото'});
    }

    const criteria = fields.map((field, index) => (
        <Criteria
            key={field.name}
            name={field.name}
            label={field.label}
            index={index}
            multipleSelect={scaleOptions.multipleSelect}
            disabled={hotkeyEventsDisabled}
        />
    ));

    const dependencies = [...fields.map(({ name }) => [name]), [VERDICT_OTHER]];

    const InputComponent = scaleOptions.multipleSelect ? withHotkey(Checkbox) : withHotkey(Radio);

    const handleDocumentFieldsChange = (fields) => onFieldsChange(convertFormFieldsToValues(fields));

    const getResetData = () => {
        const resetData = {};
        const keys = Object.keys(fieldsData);

        keys.forEach((key) => {
            resetData[key] = null;
        });

        return resetData;
    };

    const handleVerdictChange = (values) => {
        if (HIDE_AUTOFILL_VERDICTS.includes(values[0].name[0])) {
            const resetData = getResetData();

            documentForm.setFieldsValue(resetData);
            onFieldsChange(resetData);
        }

        if (!values[0]) {
            return;
        }

        resetTaskError();

        const { name, value } = values[0];
        const namePath = name.join('.');

        if (namePath === VERDICT_OK) {
            setIsStrictValidation(value);
        }

        if (namePath === VERDICT_OK && value) {
            verdictsForm.setFieldsValue({
                ...fields.reduce((results, { name: field }) => ({ ...results, [field]: false }), {}),
                [VERDICT_OTHER]: false,
            });

            return onVerdictChange({ verdicts: { [documentType]: [VERDICT_OK] }, comments: { [document]: null } });
        }

        if (fields.some(({ name }) => name === namePath) || namePath === VERDICT_OTHER) {
            verdictsForm.setFieldsValue({ [VERDICT_OK]: false });

            if (!scaleOptions.multipleSelect) {
                verdictsForm.setFieldsValue({
                    ...fields.reduce((results, { name: field }) => ({ ...results, [field]: false }), {}),
                    [VERDICT_OTHER]: false,
                    [namePath]: value,
                });
            }

            let results = verdicts.filter((verdict) => verdict !== VERDICT_OK && verdict !== namePath);

            results = value ? (scaleOptions.multipleSelect ? [...results, namePath] : [namePath]) : results;

            return onVerdictChange({ verdicts: { [documentType]: results } });
        }

        if (namePath === 'description') {
            onVerdictChange({ comments: { [documentType]: value } });
        }
    };

    const handleCommentKeyDown = (e) => {
        const hasModifier = e.ctrlKey || e.metaKey;

        if (e.key !== 'Enter') {
            return;
        }

        if (!hasModifier) {
            return e.preventDefault();
        }

        commentRef.current?.blur();
    };

    const renderSubmitter = () => (
        <section className="document__submit-overlay">
            <p>{translate('common_submitting')}</p>
            <Spin />
        </section>
    );

    return (
        // eslint-disable-next-line camelcase
        <section className={classNames('document', { document_verified: isVerified })}>
            {submitting && renderSubmitter()}

            <Typography.Title level={4} type="secondary" className="document__header">
                {translate(documentType)}
                <Hint title={translate(isVerified ? 'document_verified_hint' : 'document_not_verified_hint')}>
                    {isVerified ? <CheckCircleOutlined /> : <QuestionCircleOutlined />}
                </Hint>
            </Typography.Title>

            <section className="document__fields">
                <Form
                    name={documentType}
                    form={documentForm}
                    labelAlign="left"
                    initialValues={initialValues}
                    onFieldsChange={handleDocumentFieldsChange}
                    colon={false}
                    {...formLayout}
                >
                    {isForeign && <CountryPicker />}

                    <Form.Item noStyle shouldUpdate>
                        {({ getFieldValue }) =>
                            !isForeign || (isForeign && !['N/A', 'N/R'].includes(getFieldValue('country'))) ? (
                                <>
                                    {documentType === DOCUMENT_TYPES.passportBio && (
                                        <PassportBio
                                            availableFields={availableFields}
                                            isStrictValidation={isStrictValidation}
                                            isForeign={isForeign && getFieldValue('country') !== 'RUS'}
                                        />
                                    )}

                                    {documentType === DOCUMENT_TYPES.passportReg && (
                                        <PassportReg
                                            availableFields={availableFields}
                                            isStrictValidation={isStrictValidation}
                                            isForeign={isForeign && getFieldValue('country') !== 'RUS'}
                                            onAddressSearch={onAddressSearch}
                                            formInstance={documentForm}
                                        />
                                    )}

                                    {documentType === DOCUMENT_TYPES.licenseFront && (
                                        <LicenseFront
                                            availableFields={availableFields}
                                            isStrictValidation={isStrictValidation}
                                            isForeign={isForeign && getFieldValue('country') !== 'RUS'}
                                        />
                                    )}

                                    {documentType === DOCUMENT_TYPES.licenseBack && (
                                        <LicenseBack
                                            availableFields={availableFields}
                                            isStrictValidation={isStrictValidation}
                                            isForeign={isForeign && getFieldValue('country') !== 'RUS'}
                                        />
                                    )}
                                </>
                            ) : null
                        }
                    </Form.Item>
                </Form>
            </section>

            <Divider />

            <Form
                name={verdictsFormName}
                form={verdictsForm}
                onFieldsChange={handleVerdictChange}
                labelAlign="left"
                hideRequiredMark
                colon={false}
                {...formLayout}
            >
                {criteria}

                {scaleOptions.otherWithComment && (
                    <>
                        <Form.Item className="ant-form-item_compact" name={VERDICT_OTHER} valuePropName="checked">
                            <InputComponent hotKey={{ code: 'KeyQ', label: 'Q' }} disabled={hotkeyEventsDisabled}>
                                {translate('verdict_other')}
                            </InputComponent>
                        </Form.Item>
                        <Form.Item
                            noStyle
                            shouldUpdate={(prevValues, currentValues) =>
                                prevValues[VERDICT_OTHER] !== currentValues[VERDICT_OTHER]
                            }
                        >
                            {({ getFieldValue }) =>
                                getFieldValue([VERDICT_OTHER]) && (
                                    <Form.Item
                                        name="description"
                                        rules={[
                                            ({ getFieldValue }) => ({
                                                validator(_, value) {
                                                    const isRequired = getFieldValue([VERDICT_OTHER]);

                                                    if (!value && isRequired) {
                                                        return Promise.reject(
                                                            translate('error_validation_description'),
                                                        );
                                                    }

                                                    return Promise.resolve();
                                                },
                                            }),
                                        ]}
                                        preserve={false}
                                    >
                                        <Input
                                            ref={commentRef}
                                            autoFocus
                                            className="sidebar__description"
                                            placeholder={translate('verdict_description_placeholder')}
                                            onKeyDown={handleCommentKeyDown}
                                            disabled={hotkeyEventsDisabled}
                                        />
                                    </Form.Item>
                                )
                            }
                        </Form.Item>
                    </>
                )}

                <Form.Item
                    name={VERDICT_OK}
                    dependencies={dependencies}
                    valuePropName="checked"
                    validateStatus={taskError ? 'error' : null}
                    help={taskError}
                    rules={[
                        ({ getFieldsValue }) => ({
                            validator(_, value) {
                                const formValues = getFieldsValue();

                                if (!value && Object.keys(formValues).every((field) => !formValues[field])) {
                                    return Promise.reject(translate('error_validation_criteria'));
                                }
                                return Promise.resolve();
                            },
                        }),
                    ]}
                >
                    <InputComponent hotKey={{ code: 'KeyW', label: 'W' }} disabled={hotkeyEventsDisabled}>
                        {translate('verdict_ok')}
                    </InputComponent>
                </Form.Item>
            </Form>
        </section>
    );
};

DocumentView.propTypes = {
    documentType: PropTypes.oneOf(Object.values(DOCUMENT_TYPES)),
    isVerified: PropTypes.bool,
    formsData: PropTypes.shape({ [PropTypes.string.isRequired]: PropTypes.any }),
    fieldsData: PropTypes.shape({ [PropTypes.string.isRequired]: PropTypes.any }),
    onFieldsChange: PropTypes.func.isRequired,
    facts: PropTypes.object.isRequired,
    verdicts: PropTypes.arrayOf(PropTypes.string),
    onVerdictChange: PropTypes.func.isRequired,
    scaleOptions: PropTypes.shape({
        multipleSelect: PropTypes.bool,
        otherWithComment: PropTypes.bool,
    }),
    taskError: PropTypes.string,
    resetTaskError: PropTypes.func.isRequired,
    activeDocumentIndex: PropTypes.number,
    documentIndex: PropTypes.number,
    onAddressSearch: PropTypes.func.isRequired,
    isForeign: PropTypes.bool,
    submitting: PropTypes.bool,
};
