import React from 'react';
import PropTypes from 'prop-types';
import { Form, Checkbox, Select } from 'antd';
import { CalendarOutlined } from '@ant-design/icons';
import { translate } from '../../../utils/translate';
import { LICENSE_CATEGORIES, MIN_ALLOWED_BIRTH_YEAR } from '../../../domain/documents';
import { validateDate } from '../validators';
import { TextField } from '../TextField';
import { Hint } from '../../Utility/Hint';

const MIN_EXPERIENCE_YEAR = 1945;

export const LicenseBack = ({ isStrictValidation, availableFields, isForeign }) => {
    const renderField = (fieldName, index) => {
        const tabIndex = index + 1;

        switch (fieldName) {
            case 'number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="number"
                        placeholder={translate('field_license_number_placeholder')}
                        required={false}
                        pattern={isForeign ? /^[0-9А-Яа-яa-zA-Z]+$/ : /^[0-9А-Яа-яa-zA-Z]{10}$/}
                    />
                );

            case 'categories':
                return (
                    <Form.Item
                        key={fieldName}
                        label={translate('field_categories')}
                        name="categories"
                        rules={[
                            {
                                required: isStrictValidation,
                                pattern: /^[B-]$/,
                                message: translate('error_field_required'),
                            },
                        ]}
                    >
                        <Select placeholder={translate('field_categories_placeholder')} tabIndex={tabIndex} allowClear>
                            {Object.values(LICENSE_CATEGORIES).map((category) => (
                                <Select.Option key={category} value={category}>
                                    {translate(`license_category_${category}`)}
                                </Select.Option>
                            ))}
                        </Select>
                    </Form.Item>
                );

            case 'categories_b_valid_from_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="categories_b_valid_from_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_EXPERIENCE_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'categories_b_valid_to_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="categories_b_valid_to_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0) },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'special_category_b_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="special_category_b_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_EXPERIENCE_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'prev_licence_issue_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="prev_licence_issue_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'prev_licence_number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="prev_licence_number"
                        placeholder={translate('field_license_number_placeholder')}
                        required={false}
                        pattern={isForeign ? /^[0-9А-Яа-яa-zA-Z]+$/ : /^[0-9А-Яа-яa-zA-Z]{9,10}$/}
                    />
                );

            case 'experience_from':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="experience_from"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_EXPERIENCE_YEAR, 0) },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'has_at_mark':
                return (
                    <Hint title={translate('field_has_at_mark_hint')} placement="topLeft" key={fieldName}>
                        <Form.Item label={translate('field_has_at_mark')} name="has_at_mark" valuePropName="checked">
                            <Checkbox tabIndex={tabIndex} />
                        </Form.Item>
                    </Hint>
                );
        }
    };

    return availableFields.map(renderField);
};

LicenseBack.propTypes = {
    isStrictValidation: PropTypes.bool,
    isForeign: PropTypes.bool,
    availableFields: PropTypes.arrayOf(PropTypes.string.isRequired).isRequired,
};
