import React, { useRef } from 'react';
import PropTypes from 'prop-types';
import { Form, Select } from 'antd';
import { CalendarOutlined } from '@ant-design/icons';
import { translate } from '../../../utils/translate';
import { LICENSE_CATEGORIES, MIN_ALLOWED_BIRTH_YEAR } from '../../../domain/documents';
import { validateDate } from '../validators';
import { TextField } from '../TextField';

const MIN_EXPERIENCE_YEAR = 1945;

export const LicenseFront = ({ isStrictValidation, availableFields, isForeign }) => {
    const categoriesRef = useRef(null);

    const renderField = (fieldName, index) => {
        const tabIndex = index + 1;
        const optionalHint = ['middle_name'].includes(fieldName) ? translate('field_optional_value_hint') : null;

        switch (fieldName) {
            case 'last_name':
            case 'first_name':
            case 'middle_name':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name={fieldName}
                        placeholder={translate(`field_${fieldName}_placeholder`)}
                        hint={optionalHint}
                        required={isStrictValidation}
                        pattern={isForeign ? /^[A-Za-z-\s']+$/ : /^[А-Яа-я-\s]+$/}
                    />
                );

            case 'birth_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="birth_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={isStrictValidation}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="number"
                        placeholder={translate('field_license_number_placeholder')}
                        required={isStrictValidation}
                        pattern={isForeign ? /^[0-9A-Za-zА-Яа-я]+$/ : /^[0-9А-Яа-яa-zA-Z]{10}$/}
                    />
                );

            case 'prev_licence_number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="prev_licence_number"
                        placeholder={translate('field_license_number_placeholder')}
                        required={false}
                        pattern={isForeign ? /^[0-9А-Яа-яa-zA-Z]+$/ : /^[0-9А-Яа-яa-zA-Z]{9,10}$/}
                    />
                );

            case 'categories':
                return (
                    <Form.Item
                        key={fieldName}
                        label={translate('field_categories')}
                        name="categories"
                        rules={[
                            {
                                required: isStrictValidation,
                                pattern: /^[B-]$/,
                                message: translate('error_field_required'),
                            },
                        ]}
                    >
                        <Select
                            placeholder={translate('field_categories_placeholder')}
                            tabIndex={tabIndex}
                            allowClear
                            showSearch
                            optionFilterProp="children"
                            showAction={['click', 'focus']}
                            onFocus={() => categoriesRef.current.focus()}
                            ref={categoriesRef}
                        >
                            {Object.values(LICENSE_CATEGORIES).map((category) => (
                                <Select.Option key={category} value={category}>
                                    {translate(`license_category_${category}`)}
                                </Select.Option>
                            ))}
                        </Select>
                    </Form.Item>
                );

            case 'categories_b_valid_to_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="categories_b_valid_to_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={isStrictValidation}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0) },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'issued_by':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name={fieldName}
                        placeholder={translate('lf_issued_by_placeholder')}
                        required={isStrictValidation}
                        pattern={/^[А-Яа-я.\d\s]+$/}
                    />
                );

            case 'special_category_b_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="special_category_b_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_EXPERIENCE_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'experience_from':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="experience_from"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_EXPERIENCE_YEAR, 0) },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'issue_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="issue_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={isStrictValidation}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'prev_licence_issue_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="prev_licence_issue_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );
        }
    };

    return availableFields.map(renderField);
};

LicenseFront.propTypes = {
    isStrictValidation: PropTypes.bool,
    isForeign: PropTypes.bool,
    availableFields: PropTypes.arrayOf(PropTypes.string.isRequired).isRequired,
};
