import React, { useMemo, useRef } from 'react';
import PropTypes from 'prop-types';
import { Form, Input, Select } from 'antd';
import { CalendarOutlined } from '@ant-design/icons';
import { i18next, translate } from '../../../utils/translate';
import { GENDERS, MIN_ALLOWED_BIRTH_YEAR } from '../../../domain/documents';
import { COUNTRIES, EXTRA_COUNTRIES } from '../../../domain/countries';
import { validateDate } from '../validators';
import { TextField } from '../TextField';

export const PassportBio = ({ isStrictValidation, availableFields, isForeign }) => {
    const genderRef = useRef(null);
    const citizenshipRef = useRef(null);

    const locale = i18next.language;
    const countries = useMemo(
        () =>
            [...EXTRA_COUNTRIES, ...COUNTRIES].map((country) => ({
                value: country.altValue.toUpperCase(),
                label: country[locale].toUpperCase(),
            })),
        [],
    );

    const renderField = (fieldName, index) => {
        const tabIndex = index + 1;
        const optionalHint = ['birth_place', 'middle_name'].includes(fieldName)
            ? translate('field_optional_value_hint')
            : null;

        switch (fieldName) {
            case 'last_name':
            case 'first_name':
            case 'middle_name':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name={fieldName}
                        placeholder={translate(`field_${fieldName}_placeholder`)}
                        hint={optionalHint}
                        required={isStrictValidation}
                        pattern={isForeign ? /^[A-Za-z-\s']+$/ : /^[А-Яа-я-\s]+$/}
                    />
                );

            case 'gender':
                return (
                    <Form.Item
                        key={fieldName}
                        label={translate('field_gender')}
                        name="gender"
                        rules={[
                            {
                                required: isStrictValidation,
                                message: translate('error_field_required'),
                            },
                        ]}
                    >
                        <Select
                            placeholder={translate('field_gender_placeholder')}
                            tabIndex={tabIndex}
                            allowClear
                            showAction={['click', 'focus']}
                            onFocus={() => genderRef.current.focus()}
                            ref={genderRef}
                        >
                            {Object.values(GENDERS).map((gender) => (
                                <Select.Option key={gender} value={gender}>
                                    {translate(`gender_${gender}`)}
                                </Select.Option>
                            ))}
                        </Select>
                    </Form.Item>
                );

            case 'citizenship':
                return (
                    <Form.Item
                        key={fieldName}
                        label={translate('field_citizenship')}
                        name="citizenship"
                        rules={[
                            {
                                required: isStrictValidation,
                                message: translate('error_field_required'),
                            },
                        ]}
                    >
                        <Select
                            placeholder={translate('field_citizenship_placeholder')}
                            tabIndex={tabIndex}
                            options={countries}
                            showSearch
                            allowClear
                            optionFilterProp="label"
                            showAction={['click', 'focus']}
                            onFocus={() => citizenshipRef.current.focus()}
                            ref={citizenshipRef}
                        />
                    </Form.Item>
                );

            case 'birth_place':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="birth_place"
                        hint={optionalHint}
                        required={isStrictValidation}
                        pattern={isForeign ? /^[()"\\/0-9A-Za-z-\s.№']+$/ : /^[()"\\/0-9А-Яа-я-\s.№]+$/}
                    />
                );

            case 'birth_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="birth_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={isStrictValidation}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="number"
                        placeholder={translate('field_passport_number_placeholder')}
                        required={isStrictValidation}
                        pattern={isForeign ? /^[0-9A-Za-zА-Яа-я]+$/ : /^[0-9А-Яа-я]{8,10}$/}
                    />
                );

            case 'issue_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="issue_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={isStrictValidation}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );

            case 'issued_by':
                return (
                    <Form.Item
                        key={fieldName}
                        name={fieldName}
                        label={translate('field_issued_by')}
                        preserve={false}
                        rules={[
                            {
                                pattern: /^[А-Яа-я.\d\s-—()'"/№]+$/,
                                message: translate('error_wrong_format'),
                            },
                            {
                                required: isStrictValidation,
                                message: translate('error_field_value_required'),
                            },
                        ]}
                    >
                        <Input.TextArea
                            allowClear
                            placeholder={translate('pb_issued_by_placeholder')}
                            tabIndex={tabIndex}
                            onPressEnter={(e) => { e.preventDefault(); }}
                        />
                    </Form.Item>
                );

            case 'subdivision_code':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="subdivision_code"
                        placeholder={translate('field_subdivision_code_placeholder')}
                        required={isStrictValidation && !isForeign}
                        pattern={/^[0-9-]{7}$/}
                        mask="111-111"
                    />
                );

            case 'expiration_date':
                return (
                    isForeign && (
                        <TextField
                            key={fieldName}
                            tabIndex={tabIndex}
                            name="expiration_date"
                            placeholder={translate('field_date_placeholder')}
                            addonBefore={<CalendarOutlined />}
                            required={isStrictValidation}
                            extraRules={[
                                {
                                    validator: validateDate,
                                    range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0), to: new Date() },
                                    message: translate('error_wrong_date_format'),
                                },
                            ]}
                            mask="1111-11-11"
                        />
                    )
                );
        }
    };

    return availableFields.map(renderField);
};

PassportBio.propTypes = {
    isStrictValidation: PropTypes.bool,
    isForeign: PropTypes.bool,
    availableFields: PropTypes.arrayOf(PropTypes.string.isRequired).isRequired,
};
