import React, { Fragment, useCallback, useState } from 'react';
import PropTypes from 'prop-types';
import { Form, Select } from 'antd';
import { CalendarOutlined } from '@ant-design/icons';
import { translate } from '../../../utils/translate';
import { debounce } from '../../../utils/helpers';
import { MIN_ALLOWED_BIRTH_YEAR } from '../../../domain/documents';
import { TextField } from '../TextField';
import { validateDate } from '../validators';

const DEBOUNCE_MS = 300;

export const PassportReg = ({ availableFields, onAddressSearch, formInstance }) => {
    const [addressSuggestOptions, setAddressSuggestOptions] = useState([]);
    const [isAddressLoading, setIsAddressLoading] = useState(false);

    const fillableFields = ['region', 'locality', 'street'].filter((fieldName) => availableFields.includes(fieldName));

    const handleAddressSearch = useCallback(
        debounce(async (query) => {
            setIsAddressLoading(true);
            setAddressSuggestOptions([]);

            try {
                const results = await onAddressSearch(query);

                setAddressSuggestOptions(results);
            } catch {
                setAddressSuggestOptions([]);
            } finally {
                setIsAddressLoading(false);
            }
        }, DEBOUNCE_MS),
        [],
    );

    const handleAddressSelect = (_, { components }) => {
        const preparedComponents = components.map(({ kind, name }) => ({
            kind: (kind || '').toLowerCase() === 'province' ? 'region' : kind,
            name: name.replace(/[ё]/gi, 'е'),
        }));

        fillableFields.forEach((field) => {
            const component = preparedComponents.filter(({ kind }) => (kind || '').toLowerCase() === field);

            if (component.length) {
                // geo suggests returns dupes for province for some weird reason
                const value = component[component.length - 1].name;

                formInstance.setFieldsValue({ [field]: value });
            }
        });
    };

    const renderField = (fieldName, index) => {
        const tabIndex = index + 1;

        switch (fieldName) {
            case 'region':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="region"
                        placeholder={translate('field_region_placeholder')}
                        required={false}
                        pattern={/^[А-Яа-я-.\s]+$/}
                    />
                );

            case 'area':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="area"
                        placeholder={translate('field_area_placeholder')}
                        required={false}
                        pattern={/^[0-9А-Яа-я-\s]+$/}
                    />
                );

            case 'locality':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="locality"
                        placeholder={translate('field_locality_placeholder')}
                        required={false}
                        pattern={/^[0-9А-Яа-я-./\s]+$/}
                    />
                );

            case 'street':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="street"
                        placeholder={translate('field_street_placeholder')}
                        required={false}
                        pattern={/^[0-9А-Яа-я-№/.\s]+$/}
                    />
                );

            case 'house':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="house"
                        placeholder={translate('field_house_placeholder')}
                        required={false}
                        pattern={/^[0-9А-Яа-я-/,]+$/}
                    />
                );

            case 'housing':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="housing"
                        placeholder={translate('field_housing_placeholder')}
                        required={false}
                        pattern={/^[0-9А-Яа-я-/.]+$/}
                    />
                );

            case 'registration_letter':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="registration_letter"
                        placeholder={translate('field_registration_letter_placeholder')}
                        required={false}
                        pattern={/^[а-яА-Я0-9]+$/}
                    />
                );

            case 'apartment':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="apartment"
                        placeholder={translate('field_apartment_placeholder')}
                        required={false}
                        pattern={/^[а-яА-Я0-9-/,]+$/}
                    />
                );

            case 'registration_expiration_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="registration_expiration_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        required={false}
                        extraRules={[
                            {
                                validator: validateDate,
                                range: { from: new Date(MIN_ALLOWED_BIRTH_YEAR, 0) },
                                message: translate('error_wrong_date_format'),
                            },
                        ]}
                        mask="1111-11-11"
                    />
                );
        }
    };

    return (
        <Fragment>
            <Form.Item label={translate('field_search_suggest')}>
                <Select
                    autoFocus
                    allowClear
                    showSearch
                    placeholder={translate('field_search_suggest_placeholder')}
                    defaultActiveFirstOption={false}
                    filterOption={false}
                    onSearch={handleAddressSearch}
                    onSelect={handleAddressSelect}
                    options={addressSuggestOptions.map(({ address, components }) => ({ value: address, components }))}
                    loading={isAddressLoading}
                />
            </Form.Item>

            {availableFields.map(renderField)}
        </Fragment>
    );
};

PassportReg.propTypes = {
    availableFields: PropTypes.arrayOf(PropTypes.string.isRequired).isRequired,
    onAddressSearch: PropTypes.func.isRequired,
    formInstance: PropTypes.object,
};
