import React, { useState, useMemo } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { i18next } from '../../utils/translate';
import { translate } from '../../utils/translate';
import { VERDICTS } from '../../domain/documents';
import { Descriptions, Popover, Tag } from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import './History.less';

const tagColors = {
    [VERDICTS.notFraud]: 'green',
    [VERDICTS.maybeFraud]: 'orange',
    [VERDICTS.definitelyFraud]: 'red',
};

export const History = ({ meta, defaultFraudReasons }) => {
    const { is_fraud: isFraud, fraud_reasons: fraudReasons, comment } = meta;
    const [isVisible, setIsVisible] = useState(isFraud !== VERDICTS.notFraud);
    const reasons = defaultFraudReasons[i18next.language][isFraud];

    const handleVisibilityChange = (isVisible) => setIsVisible(isVisible);

    const content = useMemo(
        () => (
            <Descriptions column={1} size="small">
                <Descriptions.Item label={translate('verdict_label')}>
                    <Tag color={tagColors[isFraud]}>{translate(`verdict_${isFraud}`)}</Tag>
                </Descriptions.Item>
                {fraudReasons && (
                    <Descriptions.Item label={translate('fraud_reasons_label')}>
                        {fraudReasons.map((reason) => (
                            <Tag key={reason}>{reasons[reason]}</Tag>
                        ))}
                    </Descriptions.Item>
                )}
                {comment && <Descriptions.Item label={translate('history_comment')}>{comment}</Descriptions.Item>}
            </Descriptions>
        ),
        [],
    );

    return (
        <div className={classNames('history', `history__${isFraud}`)}>
            <Popover
                overlayClassName="history__content"
                content={content}
                title={translate('history_title')}
                trigger="click"
                placement="leftTop"
                visible={isVisible}
                onVisibleChange={handleVisibilityChange}
            >
                <ExclamationCircleOutlined />
            </Popover>
        </div>
    );
};

History.propTypes = {
    defaultFraudReasons: PropTypes.shape({ ru: PropTypes.object, en: PropTypes.object }),
    meta: PropTypes.shape({
        is_fraud: PropTypes.oneOf(Object.values(VERDICTS)),
        comment: PropTypes.string,
        fraud_reasons: PropTypes.arrayOf(PropTypes.string.isRequired),
    }),
};
