/* eslint-disable camelcase */
import React, {Component} from 'react';
import PropTypes from 'prop-types';
import Storage from '../../utils/storage';
import {translate} from '../../utils/translate';
import SplitPane from 'react-split-pane';
import {Button, Result} from 'antd';
import {DOCUMENT_TYPES} from '../../domain/documents';
import {ErrorIcon} from '../Icons/ErrorIcon';
import {Async} from '../Utility/Async';
import {Spin} from '../Spin';
import {ViewerPane} from '../ViewerPane';
import {Sidebar} from '../Sidebar';
import './TaskPageLayout.less';
import {initErrorCounter, logError} from "../../error-logger/errorLog"
import {store} from "../../domain/store";
import {STATUS_401} from "../../infrastructure/errors/ServerError";

// eslint-disable-next-line no-magic-numbers
const EXPIRATION_TIME = 365 * 24 * 60 * 60 * 1000;
const DEFAULT_PANE_SIZE = 710;

export class TaskPageLayoutView extends Component {
    static propTypes = {
        preloadData: PropTypes.func.isRequired,
        secret: PropTypes.string,
        documentTypes: PropTypes.arrayOf(PropTypes.oneOf(Object.values(DOCUMENT_TYPES))).isRequired,
    };

    get splitSize() {
        return this.storage.getItem('driveDocsSplitSize') || DEFAULT_PANE_SIZE;
    }

    constructor(props) {
        super(props);

        this.storage = new Storage();
    }

    componentWillUnmount() {
        this.storage.destroy();
    }

    getErrorSolution(error) {
        let documentTypes = store.getState().input?.documentTypes ?? [];

        let documents = {};
        let verdicts = {};

        documentTypes.forEach(doc => {
            documents[doc] = {
                data: {
                    status: error.status,
                    message: error.message
                }
            };
            verdicts[doc] = 'YANG_ERROR';
        })

        return [{
            task_id: 0,
            output_values: {documents, verdicts}
        }]
    }

    submitError(error) {
        const {sandboxChannels, assignmentId} = this.props;
        const solutions = this.getErrorSolution(error);
        sandboxChannels.triggerOut('assignment:submit', {solutions, assignmentId});

        initErrorCounter(this.secret);
        logError(error, `Ошибка во время загрузки документов`);
    }

    handleSplitSize = (size) =>
        this.storage.setItem('driveDocsSplitSize', size, new Date().getTime() + EXPIRATION_TIME);

    renderError = (error) => {
        let isValidError = error.status !== STATUS_401;

        let extraInfo = isValidError
            ? <div>
                <div>
                    <h3>Хьюстон, у нас проблемы. Чтобы отправить данное задание с ошибкой, кликни ниже</h3>
                </div>
                <div>
                    <Button onClick={this.submitError.bind(this, error)}>
                        Вот сюда
                    </Button>
                </div>
            </div>
            : <div>Что-то не так. Попробуйте обновить страницу.</div>

        return <Result
            className="task-pane__error"
            extra={extraInfo}
            title={translate('error_loading_document')}
            subTitle={error.message}
            icon={<ErrorIcon width="100%" height="100%"/>}
        />
    }

    renderLoader = () => <Spin size={32} className="task-pane__loader"/>;

    renderMainLayout = () => (
        <div className="task-pane">
            <div className="task-pane__row">
                <SplitPane
                    primary="second"
                    split="vertical"
                    minSize={DEFAULT_PANE_SIZE}
                    defaultSize={this.splitSize}
                    onDragFinished={this.handleSplitSize}
                >
                    <main className="task-pane__main">
                        <ViewerPane/>
                    </main>

                    <aside className="task-pane__aside">
                        <Sidebar documentTypes={this.props.documentTypes}/>
                    </aside>
                </SplitPane>
            </div>
        </div>
    );

    render() {
        const {preloadData} = this.props;

        return (
            <Async promise={preloadData()} renderErrorLayout={this.renderError} renderLoadingLayout={this.renderLoader}>
                {() => this.renderMainLayout()}
            </Async>
        );
    }
}
