import { useEffect } from 'react';

const EventType = {
    keydown: 'keydown',
    wheel: 'wheel',
};

const ComboKey = {
    ctrl: 'ctrl',
    shift: 'shift',
    alt: 'alt',
};

const ALLOWED_COMBO_KEYS = Object.keys(ComboKey);
const ALLOWED_EVENTS = Object.keys(EventType);
const EDITABLE_TAGS = ['INPUT', 'TEXTAREA'];

const allComboKeysPressed = (keys, event) => {
    const comboKeys = keys.map((key) => key.toLowerCase());

    if (comboKeys.includes('ctrl') && !event.ctrlKey && !event.metaKey) {
        return false;
    }

    if (comboKeys.includes('shift') && !event.shiftKey) {
        return false;
    }

    return !(comboKeys.includes('alt') && !event.altKey);
};

const validateKeys = (keys) => {
    let errorMessage = null;

    if (!keys.length) {
        errorMessage = 'Need an actual key, not just combo keys';
    }

    if (keys.length > 1) {
        errorMessage = "Can't use multiple keys, only extra combo keys allowed";
    }

    if (errorMessage) {
        throw new Error(errorMessage);
    }

    return true;
};

const withoutComboKeys = (key) => !ALLOWED_COMBO_KEYS.includes(key.toLowerCase());

const comboKeys = (key) => ALLOWED_COMBO_KEYS.includes(key.toLowerCase());

const isSingleKeyEvent = (e) => !e.shiftKey && !e.metaKey && !e.altKey && !e.ctrlKey;

const isSingleKeyShortcut = (shortcut) => !shortcut.keys.filter(comboKeys).length;

const getKeyCode = (key) => {
    if (key.length === 1 && key.search(/[^a-zA-Z]+/) === -1) {
        return `Key${key.toUpperCase()}`;
    }

    if (key.length === 1 && typeof Number(key) === 'number') {
        return `Digit${key}`;
    }

    return key;
};

export const useHotkeys = (shortcuts, active = true, dependencies = [], eventType = 'keydown') => {
    if (!shortcuts || !shortcuts.length) {
        throw new Error('Need at least one shortcut');
    }

    if (!ALLOWED_EVENTS.includes(eventType)) {
        throw new Error('Unsupported event');
    }

    const shortcutHasPriority = (inputShortcut, event) => {
        if (shortcuts.length === 1) {
            return true;
        }

        if (isSingleKeyShortcut(inputShortcut) && isSingleKeyEvent(event)) {
            return true;
        }

        return !shortcuts.find(
            (shortcut) => allComboKeysPressed(shortcut.keys, event) && shortcut.keys.length > inputShortcut.keys.length,
        );
    };

    const callCallback = (shortcut, event) => {
        if (event instanceof KeyboardEvent) {
            const keys = shortcut.keys.filter(withoutComboKeys);
            const valid = validateKeys(keys);
            const singleKey = shortcut.keys.length === 1 && shortcut.keys[0] === event.key;

            if (!valid || !(singleKey || getKeyCode(keys[0]) === event.code)) {
                return;
            }
        }

        const writing =
            EDITABLE_TAGS.includes(event.target && event.target['tagName']) &&
            'code' in event &&
            event.code !== 'Escape' &&
            !event.ctrlKey &&
            !event.metaKey;

        const shouldExecAction =
            !shortcut.disabled &&
            !writing &&
            allComboKeysPressed(shortcut.keys, event) &&
            shortcutHasPriority(shortcut, event);

        if (shouldExecAction) {
            event.preventDefault();
            shortcut.onEvent(event);
        }
    };

    const handleKeyboardShortcuts = (e) => shortcuts.forEach((shortcut) => callCallback(shortcut, e));

    const addEventListener = () =>
        document.addEventListener(eventType, handleKeyboardShortcuts, {
            passive: false,
        });

    const removeEventListener = () => document.removeEventListener(eventType, handleKeyboardShortcuts);

    useEffect(() => {
        active ? addEventListener() : removeEventListener();

        return removeEventListener;
    }, [active, ...dependencies]);
};
