import { parse, stringify } from 'qs';
import { cacheFactory } from '../Cache';
import { AbstractREST } from './AbstractREST';
import { createProxyServerError } from '../errors/ServerError';

const DEFAULT_TTL = 5000;

export class ProxyREST extends AbstractREST {
    constructor({ baseUrl, sandboxChannel, assignmentId, origin, namespace = 'ajax', ttl = DEFAULT_TTL }) {
        super({ baseUrl });
        this._baseUrl = baseUrl;
        this._sandboxChannel = sandboxChannel;
        this._assignmentId = assignmentId;
        this._origin = origin;
        this._cache = cacheFactory({ namespace, ttl });
    }

    upperCaseCountries(data){
        if (data && (data.license_front || data.license_back || data.passport_selfie || data.passport_biographical || data.passport_registration)){
            for (const [key, value] of Object.entries(data)) {
                if (value.data && value.data.country){
                    data[key].data.country = value.data.country.toUpperCase()
                }
            }
        }
        return data;
    }

    _proxyRequest(path, options) {
        // в свежих версиях фронта Толоки url формируется автоматически
        // в этом месте мы формируем его только для отладки в TTS <= 1.1.34, так как там остался старый механизм
        const url =
            window?.location?.host === 'localhost:9000'
                ? this.getProxyRequestUrl(path.replace(this._baseUrl, ''))
                : null;

        return new Promise((resolve, reject) =>
            this._sandboxChannel
                .request('proxy', { ...options, path, url })
                .then((response) => {
                    resolve(this.upperCaseCountries(response.data));
                })
                .catch((error) => {
                    if (error.data && error.data.status === 200) {
                        resolve(this.upperCaseCountries(error.data));
                    }
                    else{
                        reject(this.upperCaseCountries(error.data));
                    }
                })
        );
    }

    _fetch(url, options) {
        var opts = null
        if (options.body){
            options.body = this.upperCaseCountries(options.body)
        }
        else{
            if (options.data){
                options.data = this.upperCaseCountries(options.data)
            }
        }
        if (options.type){
            opts = {
                type : options.type,
                contentType: 'application/json',
                data: options.body ? JSON.stringify(options.body) : options.data,
                dataType: 'json',
                processData: true,
                ...options
            };
        }
        else{
            opts = {
                type: 'GET',
                contentType: 'application/json',
                data: options.body ? JSON.stringify(options.body) : options.data,
                dataType: 'json',
                ...options
            };
        }
        return Promise.resolve(
            this._proxyRequest(this.createUrl(url), opts),
        )
            .then((data) => {
                this._cache.save(url, data);
                return data;
            })
            .catch((response) => {
                throw createProxyServerError(response);
            });
    }

    _parseUrl(fullUrl) {
        const [, url, query] = fullUrl.match(/([^?]*)\??(.*)/);
        const queryParams = query ? parse(query) : {};

        return { url, queryParams };
    }

    getProxyRequestUrl(path, data = {}) {
        const origin = this._origin;
        const urlPath = path.replace(/^\/?/, '');

        return (
            [origin, 'api/proxy', `${this._baseUrl}/${urlPath}`].join('/') +
            stringify({ ...data, assignmentId: this._assignmentId }, { addQueryPrefix: true })
        );
    }

    // eslint-disable-next-line no-unused-vars
    async create(url, data) {
        console.warn('ProxyREST.create is not implemented');
    }

    async read(urlWithParams, options = {}) {
        const { url, queryParams } = this._parseUrl(urlWithParams);

        return this._cachedFetch(url, {
            ...options,
            data: queryParams,
        });
    }

    // eslint-disable-next-line no-unused-vars
    async update(url, data) {
        console.warn('ProxyREST.update is not implemented');
    }

    // eslint-disable-next-line no-unused-vars
    async delete(url) {
        console.warn('ProxyREST.delete is not implemented');
    }

    async request({ url, method, body }) {
        return this._fetch(url, {
            type: method,
            body,
        });
    }
}