import {store} from "../domain/store";
import {STATUS_404, STATUS_500} from "../infrastructure/errors/ServerError";
import {serviceLocator} from '../infrastructure/serviceLocator';
import {stringify} from "qs";

export default class DriveSBSAssignment extends window.TolokaAssignment {
    /**
     * Provides solutions to the app if they are valid.
     *
     * @param {Function} strategy
     * @param {Function} errorCallback - called if validation errors
     */
    provideSolutions(
        strategy = function (solutions) {
            this.getSandboxChannel().triggerOut('assignment:submit', { solutions, assignmentId: this.getId() });
        },
        // eslint-disable-next-line no-unused-vars
        errorCallback = function () {},
    ) {
        const solutions = this.getTaskSuite().getSolutions();

        Promise.resolve(this.getTaskSuite().validate(solutions)).then(async (errors) => {
            if (!errors) {
                let {taskType} = store.getState()?.input;

                if (taskType === 'registration_on_chat') {
                    await this.addToQueueRequest(strategy, solutions);
                } else {
                    try {
                        const { sendToBackend } = this.getTaskSuite().getScaleOptions();
                        await Promise.all(
                            sendToBackend
                                ? this.getTaskSuite().getTasks().map((task) => task.controller.submitResults())
                                : []
                        ).then(async () => {
                            if (taskType === 'reissue_license') {
                                this.setStatus(strategy);
                            } else {
                                strategy.call(this, solutions);
                            }
                        });
                    } catch (error) {
                        if ([STATUS_404, STATUS_500].includes(error?.status)) {
                            let solution = getErrorSolution(solutions, error);
                            let {taskType} = store.getState()?.input;

                            if (taskType === 'reissue_license') {
                                this.setStatus(strategy, error);
                            } else {
                                strategy.call(this, solution);
                            }
                        }
                    }
                }
            } else {
                this.getSandboxChannel().triggerOut('assignment:validation:fail', errors);
            }
        });
    }

    setStatus(strategy, errorSolution) {
        let {userId, secret: secretId} = store.getState()?.input;
        let verdict = Object.entries(store.getState()?.solution?.verdicts ?? {})?.[0];
        const statusTypes = this.getTaskSuite().getSetStatusType();
        const finalVerdict = [statusTypes[verdict[0]], verdict[1][0]];

        const solutions = errorSolution
            ? getErrorSolution(this.getTaskSuite().getSolutions(), errorSolution)
            : this.getTaskSuite().getSolutions();

        setStatus(userId, finalVerdict, secretId)
            .then(() => {
                this.setSendServerStatus(solutions, false);
                strategy.call(this, solutions);
            })
            .catch(error => {
                this.setSendServerStatus(solutions, true);
                strategy.call(this, solutions);
            })
    }

    async addToQueueRequest(strategy, solutions) {
        const { secret } = store.getState().input;
        const queryParams = stringify({ secretId: secret }, { addQueryPrefix: true });
        const statusTypes = this.getTaskSuite().getSetStatusType();

        const verdicts = solutions[0].output_values.verdicts;
        const other = solutions[0].output_values.other;
        const documents = solutions[0].output_values.documents;
        const output_values = {};
        output_values.verdicts = Object.keys(verdicts)
            .reduce((prev, curr) => {
                prev[statusTypes[curr]] = verdicts[curr];
                return prev;
            }, {});

        solutions[0].output_values = { documents, other, ...output_values };

        await serviceLocator.DataREST.request({
            url: `assignment/queue${queryParams}`,
            method: 'POST',
        }).then(() => {
            this.setSendServerStatus(solutions, false);
            strategy.call(this, solutions);
        }).catch((error) => {
            const SUCCESS_STATUS = 200;

            this.setSendServerStatus(solutions, error?.status && error?.status !== SUCCESS_STATUS);
            strategy.call(this, solutions);
        });
    }

    setSendServerStatus(solutions, isError) {
        solutions[0].output_values.send_server_status = isError ? 'error' : 'ok';
    }
}

async function setStatus(userId, verdict, secretId) {
    const queryParams = stringify({ secretId }, { addQueryPrefix: true });

    await serviceLocator.ChecksREST.request({
        url:`${queryParams}`,
        method: 'POST',
        body: {
            user_id: userId,
            type: verdict[0],
            status: verdict[1] ?? 'can not get status',
        }
    })
}

function getErrorSolution(solutions, error) {
    let document_types = Object.keys(solutions[0].output_values.documents);

    let verdicts = {};
    let other = {};
    let documents = {};

    if (document_types.length) {
        document_types.forEach((type) => {
            verdicts[type] = ['YANG_ERROR'];
            other[type] = '';
            documents[type] = {
                data: error.commonErrorData ?? error.message,
            }
        })
    } else {
        verdicts = 'YANG_ERROR';
    }

    return [{
        ...solutions[0],
        output_values: {
            documents,
            verdicts,
            other
        }
    }]
}
