import { createElement } from 'react';
import { render, unmountComponentAtNode } from 'react-dom';
import { initStore, resetStore, store } from '../domain/store';
import { setTaskError } from '../domain/store/actions/ui';
import { serviceLocator } from '../infrastructure/serviceLocator';
import { submitResultsUsecase } from '../usecases/submitResultsUsecase';
import { convertOutputUsecase } from '../usecases/convertOutputUsecase';
import { TaskView } from '../components/TaskView';

export class TaskPageController {
    constructor({ element, input, facts, confidenceThresholds, scaleOptions }) {
        this.element = element;
        this.input = input;
        this.facts = facts;
        this.confidenceThresholds = confidenceThresholds;
        this.scaleOptions = scaleOptions;
    }

    get view() {
        return this._view;
    }

    createView() {
        this._view = render(createElement(TaskView), this.element);
    }

    getResults() {
        const { verdicts, comments: other } = store.getState().solution;

        return {
            verdicts,
            other,
            documents: this.scaleOptions.sendToOutputValues
                ? convertOutputUsecase(store.getState().solution.documents)
                : {},
        };
    }

    async submitResults() {
        const { documents, verdicts } = store.getState().solution;

        await submitResultsUsecase(documents, verdicts);
    }

    run() {
        initStore(this.input, this.facts, this.confidenceThresholds, this.scaleOptions);
        this.createView();
    }

    destroy() {
        unmountComponentAtNode(this.element);

        if (this.element.parentNode) {
            this.element.parentNode.removeChild(this.element);
        }

        resetStore();
    }

    showTaskError(message) {
        store.dispatch(setTaskError(message));
    }

    async validateAndCollect() {
        return serviceLocator.validateAndCollect();
    }
}
