/* eslint-disable camelcase */
import { serviceLocator } from '../infrastructure/serviceLocator';
import { dispatch } from '../domain/store';
import { setInitialDocumentsData, setInitialFormsData } from '../domain/store/actions/documents';
import { initSolution } from '../domain/store/actions/solution';
import { isEmptyObject, sortObjectKeys } from '../utils/fn';
import { stringify } from 'qs';
import { DEFAULT_FIELDS_ORDER, DOCUMENT_TYPES } from '../domain/documents';
import { setOnlyAvailableFields } from '../utils/setOnlyAvailableFields';

export const getDocumentsUsecase = async ({
    secret,
    documentTypes,
    confidenceThresholds,
    availableFields,
    isForeign,
    isCustomField,
    outdata
}) => {
    const queryParams = stringify({ secretId: secret }, { addQueryPrefix: true });

    let assignmentDocuments = await serviceLocator.DataREST.read(`assignment/user_info${queryParams}`);

    // pre-populate documents dictionary
    const defaultDocuments = Object.values(DOCUMENT_TYPES).reduce(
        (dictionary, type) => ({ ...dictionary, [type]: {} }),
        {},
    );

    let documents = { ...defaultDocuments };

    if (isCustomField) {
        documents = {
            ...documents,
            ...outdata
        }
    } else {
        documents = {
            ...documents,
            ...assignmentDocuments
        }
    }

    // 1. strip unavailable document types
    documents = Object.entries(documents).reduce(
        (results, [document, values]) =>
            documentTypes.includes(document) ? { ...results, [document]: { ...values } } : results,
        {},
    );

    documents = setOnlyAvailableFields(documents, availableFields);

    dispatch(setInitialDocumentsData(documents));

    // 2. if available fields is set, ignore confidence thresholds
    if (!isEmptyObject(availableFields)) {
        documents = Object.entries(documents).reduce((results, [document, { data }]) => {
            if (isEmptyObject(data) || !availableFields[document]) {
                return {
                    ...results,
                    [document]: { data: { ...sortObjectKeys(data, DEFAULT_FIELDS_ORDER[document]) } },
                };
            }

            return {
                ...results,
                [document]: {
                    data: sortObjectKeys(
                        Object.entries(data).reduce((dict, [field, value]) => {
                            // do we actually need to prefill here?
                            if (field === 'country' && isForeign) {
                                return { ...dict, [field]: value };
                            }

                            if (field === 'citizenship') {
                                return { ...dict, [field]: value };
                            }

                            if (availableFields[document].includes(field)) {
                                return { ...dict, [field]: null };
                            }

                            return dict;
                        }, {}),
                        availableFields[document] || DEFAULT_FIELDS_ORDER[document],
                    ),
                },
            };
        }, {});
    }

    // 3. if no available fields is set, only display fields with low confidence
    if (isEmptyObject(availableFields)) {
        documents = Object.entries(documents).reduce((results, [document, { data, confidence }]) => {
            const thresholds = confidenceThresholds[document];

            if (isEmptyObject(confidenceThresholds) || !thresholds || isEmptyObject(confidence)) {
                return {
                    ...results,
                    [document]: { data: { ...sortObjectKeys(data, DEFAULT_FIELDS_ORDER[document]) } },
                };
            }

            return {
                ...results,
                [document]: {
                    data: sortObjectKeys(
                        Object.keys(data).reduce((dict, field) => {
                            if (field === 'country' && isForeign) {
                                return { ...dict, [field]: data[field] };
                            }

                            if (confidence[field] <= thresholds[field]) {
                                return { ...dict, [field]: null };
                            }

                            return dict;
                        }, {}),
                        DEFAULT_FIELDS_ORDER[document],
                    ),
                },
            };
        }, {});
    }

    dispatch(setInitialFormsData(documents));
    dispatch(initSolution({ documents, documentTypes }));
};
