/* eslint-disable camelcase */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { translate } from '../../utils/translate';
import { VERDICT_BAD_PHOTO, VERDICT_OK, VERDICT_OTHER } from '../../domain/constants';
import { Form, Typography, Input, Radio, Checkbox } from 'antd';
import './Sidebar.less';
import { Async } from '../Utility/Async';
import { withHotkey } from '../Utility/WithHotkey';

const MAX_HOTKEYS = 9;
const HOT_KEYS = ['T', 'Y', 'U', 'I'];

const formLayout = {
    wrapperCol: { span: 24 },
};

const Criteria = ({ name, label, index, multipleSelect, hotKey }) => {
    const newHotKey = hotKey ?? (index < MAX_HOTKEYS ? { code: `Digit${index + 1}`, label: index + 1 } : null);
    const InputComponent = multipleSelect ? withHotkey(Checkbox) : withHotkey(Radio);

    return (
        <Form.Item name={name} valuePropName="checked">
            <InputComponent hotKey={newHotKey} multiple={multipleSelect}>
                {label}
            </InputComponent>
        </Form.Item>
    );
};

export class SidebarView extends PureComponent {
    static propTypes = {
        verdicts: PropTypes.arrayOf(PropTypes.oneOfType([PropTypes.string, PropTypes.object])),
        addVerdict: PropTypes.func.isRequired,
        setVerdict: PropTypes.func.isRequired,
        removeVerdict: PropTypes.func.isRequired,
        comment: PropTypes.string,
        setComment: PropTypes.func.isRequired,
        setFormError: PropTypes.func.isRequired,
        resetTaskError: PropTypes.func.isRequired,
        preloadAgeData: PropTypes.func.isRequired,
        taskError: PropTypes.string,
        validateSignal: PropTypes.number,
        facts: PropTypes.object,
        secret: PropTypes.string,
        taskType: PropTypes.string,
        documentTypes: PropTypes.arrayOf(PropTypes.string),
    };

    formRef = React.createRef();
    sidebarRef = React.createRef();
    commentRef = React.createRef();

    constructor(props) {
        super(props);

        this.ageBlock = this.renderAge(props.secret);
    }

    /**
     * Push changes to store
     * @param {object} values
     */
    handleValuesChange = (values) => {
        const { resetTaskError, setComment, addVerdict, removeVerdict } = this.props;

        resetTaskError();

        Object.keys(values).forEach((name) => {
            if (name === 'description') {
                return setComment(values[name]);
            }

            if (values[name]) {
                addVerdict(name);
            } else {
                removeVerdict(name);
            }
        });
    };

    handleBadQualityValuesChange = (values) => {
        const { resetTaskError, setComment, facts, verdicts, setVerdict, documentTypes } = this.props;
        const fields = Object.keys(facts);

        resetTaskError();

        Object.keys(values).forEach((name) => {
            if (name === 'description') {
                return setComment(values[name]);
            }

            if (documentTypes.includes(name)) {
                let badQualityVerdicts = verdicts[0]?.[VERDICT_BAD_PHOTO]?.filter((item) => item !== name) ?? [];
                const result = values[name]
                    ? [...badQualityVerdicts, name]
                    : badQualityVerdicts;

                this.formRef.current.setFieldsValue({
                    ...this.formRef.current.getFieldValue(),
                    [VERDICT_BAD_PHOTO]: result.length ? result : true,
                });

                return setVerdict({ verdicts: [{ [VERDICT_BAD_PHOTO]: result }] });
            }

            this.formRef.current.setFieldsValue({
                ...fields.reduce((results, field) => ({ ...results, [field]: false }), {}),
                [VERDICT_OTHER]: false,
                [VERDICT_OK]: false,
                [name]: true,
            });

            let result;

            if (name === VERDICT_BAD_PHOTO) {
                const fieldValues = this.formRef.current.getFieldValue();
                const results = Object.keys(fieldValues).filter((item) => documentTypes.includes(item) && fieldValues[item]);
                result = [{ [VERDICT_BAD_PHOTO]: results }];
            } else {
                result = [name];
            }

            return setVerdict({ verdicts: result });
        });
    }

    handleCommentKeyDown = (e) => {
        const hasModifier = e.ctrlKey || e.metaKey;

        if (e.key !== 'Enter') {
            return;
        }

        if (!hasModifier) {
            return e.preventDefault();
        }

        // eslint-disable-next-line no-unused-expressions
        this.commentRef.current?.blur();
    };

    validateAll = async () => {
        let success = false;

        try {
            await this.formRef.current.validateFields();
            this.props.setFormError(false);
            success = true;
        } catch {
            this.props.setFormError(true);
        }

        return success;
    };

    componentDidMount() {
        window.requestAnimationFrame(() => this.sidebarRef.current.focus());
    }

    componentDidUpdate(prevProps) {
        if (prevProps.validateSignal !== this.props.validateSignal) {
            this.validateAll();
        }
    }

    renderAge = (secret) => {
        const { preloadAgeData } = this.props;

        return (
            <Async promise={preloadAgeData({ secret })} renderErrorLayout={() => null}>
                {(value) => (
                    <div className="sidebar__age">
                        {!!value?.passportAge && <Typography.Text type="secondary">
                            {`Возраст в паспорте: ${value.passportAge}`}
                        </Typography.Text>}
                        {!!value?.licenseAge && <Typography.Text type="secondary">
                            {`Возраст в ВУ: ${value.licenseAge}`}
                        </Typography.Text>}
                    </div>)}
            </Async>
        );
    }

    renderBadQuality = () => {
        const { facts, documentTypes, verdicts } = this.props;

        return (
            <>
                <Criteria key={VERDICT_BAD_PHOTO}
                          name={VERDICT_BAD_PHOTO}
                          label={facts[VERDICT_BAD_PHOTO]}
                          index={0} />
                <Form.Item
                    noStyle
                    shouldUpdate={(prevValues, currentValues) =>
                        prevValues[VERDICT_BAD_PHOTO] !== currentValues[VERDICT_BAD_PHOTO]
                    }
                >
                    {({ getFieldValue }) =>
                        getFieldValue([VERDICT_BAD_PHOTO]) && (
                            <Form.Item
                                name={VERDICT_BAD_PHOTO}
                                style={{ marginLeft: '25px' }}
                                rules={[
                                    ({ getFieldValue }) => ({
                                        validator(_, value) {
                                            const isRequired = getFieldValue([VERDICT_BAD_PHOTO]);

                                            if (!verdicts[0][VERDICT_BAD_PHOTO].length && isRequired) {
                                                return Promise.reject(translate('error_field_required'));
                                            }

                                            return Promise.resolve();
                                        },
                                    }),
                                ]}
                            >
                                <>
                                    {documentTypes.map((item, index) => (
                                        <Criteria key={item}
                                                  name={item}
                                                  label={translate(item)}
                                                  index={index + 1}
                                                  multipleSelect={true} />
                                    ))}
                                </>
                            </Form.Item>
                        )
                    }
                </Form.Item>
            </>
        );
    }

    render() {
        const { facts, verdicts, comment, taskError, taskType } = this.props;

        const fields = Object.keys(facts).filter(item => item !== VERDICT_BAD_PHOTO).map((name) => ({
            name,
            label: facts[name],
            value: verdicts.includes(name),
        }));
        const multiple = taskType !== 'registration_on_chat';
        const isFaceCompare = Object.keys(facts).includes(VERDICT_BAD_PHOTO);
        const InputComponent = multiple ? withHotkey(Checkbox) : withHotkey(Radio);

        const criteria = fields.map((field, index) => (
            <Criteria key={field.name}
                      name={field.name}
                      label={field.label}
                      hotKey={!multiple ? { code: `Key${HOT_KEYS[index]}`, label: HOT_KEYS[index] } : null}
                      index={multiple ? index : null}
                      multipleSelect={multiple} />
        ));

        const dependencies = [...fields.map((field) => [field.name]), [VERDICT_OTHER]];

        fields.push({ name: VERDICT_OTHER, value: verdicts.includes(VERDICT_OTHER) });
        fields.push({ name: VERDICT_OK, value: verdicts.includes(VERDICT_OK) });

        fields.push({
            name: 'description',
            value: comment,
        });
        fields.push({ name: VERDICT_BAD_PHOTO });

        return (
            <div className="sidebar" ref={this.sidebarRef} tabIndex={-1}>
                <Typography.Title level={4} className="sidebar__title">
                    {translate('common_task_title')}
                </Typography.Title>
                {this.ageBlock}
                <Form
                    {...formLayout}
                    ref={this.formRef}
                    fields={fields}
                    onValuesChange={multiple ? this.handleValuesChange : this.handleBadQualityValuesChange}
                    labelAlign="left"
                    hideRequiredMark
                >
                    {criteria}
                    {!multiple && isFaceCompare && this.renderBadQuality()}

                    <Criteria key={VERDICT_OTHER}
                              name={VERDICT_OTHER}
                              label={translate('verdict_other')}
                              hotKey={{ code: 'KeyQ', label: 'Q' }}
                              multipleSelect={multiple} />
                    <Form.Item
                        noStyle
                        shouldUpdate={(prevValues, currentValues) =>
                            prevValues[VERDICT_OTHER] !== currentValues[VERDICT_OTHER]
                        }
                    >
                        {({ getFieldValue }) =>
                            getFieldValue([VERDICT_OTHER]) && (
                                <Form.Item
                                    name="description"
                                    rules={[
                                        ({ getFieldValue }) => ({
                                            validator(_, value) {
                                                const isRequired = getFieldValue([VERDICT_OTHER]);

                                                if (!value && isRequired) {
                                                    return Promise.reject(translate('error_validation_description'));
                                                }

                                                return Promise.resolve();
                                            },
                                        }),
                                    ]}
                                >
                                    <Input
                                        ref={this.commentRef}
                                        autoFocus
                                        autoComplete={'off'}
                                        className="sidebar__description"
                                        placeholder={translate('common_description_placeholder')}
                                        onKeyDown={this.handleCommentKeyDown}
                                    />
                                </Form.Item>
                            )
                        }
                    </Form.Item>

                    <Form.Item
                        name={VERDICT_OK}
                        dependencies={dependencies}
                        valuePropName="checked"
                        validateStatus={taskError ? 'error' : null}
                        help={taskError}
                        rules={[
                            ({ getFieldsValue }) => ({
                                validator(_, value) {
                                    const formValues = getFieldsValue();

                                    if (!value && Object.keys(formValues).every((field) => !formValues[field])) {
                                        return Promise.reject(translate('error_validation_criteria'));
                                    }

                                    return Promise.resolve();
                                },
                            }),
                        ]}
                    >
                        <InputComponent hotKey={{ code: 'KeyW', label: 'W' }}>
                            {translate('verdict_ok')}
                        </InputComponent>
                    </Form.Item>
                </Form>
            </div>
        );
    }
}

Criteria.propTypes = {
    name: PropTypes.string.isRequired,
    label: PropTypes.string.isRequired,
    index: PropTypes.number,
    multipleSelect: PropTypes.bool,
    hotKey: PropTypes.object,
};
