import { sleep } from '../utils/fn';
import { isObject } from '../utils/fn';

class Store {
    setItem(key, value) {} // eslint-disable-line no-unused-vars
    getItem(key) {} // eslint-disable-line no-unused-vars
    removeItem(key) {} // eslint-disable-line no-unused-vars
    clear() {}
}

class InMemoryStore extends Store {
    constructor() {
        super();
        this.map = new Map();
    }

    setItem(key, value) {
        this.map.set(key, value);
    }

    getItem(key) {
        const value = this.map.get(key);

        return typeof value === 'undefined' ? null : this.map.get(key);
    }

    removeItem(key) {
        this.map.delete(key);
    }

    clear() {
        this.map.clear();
    }
}

class Cache {
    constructor(store, keyPrefix, ttl = -1) {
        if (!keyPrefix) throw new Error('Cache: keyPrefix should be specified');

        this.keyPrefix = keyPrefix;
        this.ttl = ttl;
        this.store = store;
    }

    async save(key, value) {
        this.set(key, value);

        if (this.ttl > -1) await sleep(this.ttl);

        this.del(key);
        return value;
    }

    set(key, value) {
        return this.store.setItem(this.createKey(key), typeof value === 'undefined' ? null : value);
    }

    get(key, defaultValue = null) {
        const tmpKey = this.createKey(key);
        const value = this.store.getItem(tmpKey);

        return value === null ? defaultValue : value;
    }

    del(key) {
        this.store.removeItem(this.createKey(key));
    }

    clear() {
        this.store.clear();
    }

    /**
     * @private
     * @param {object} key
     * @returns {string}
     */
    createKey(key) {
        if (!isObject(key)) {
            return `${this.keyPrefix}:${key}`;
        }
        const strKey = Object.keys(key)
            .sort()
            .map((name) => `${name}=${this.createKey(key[name])}`)
            .join('&');

        return `${this.keyPrefix}:${strKey}`;
    }

    /**
     * @deprecated
     * @param {string} name
     */
    remove(name) {
        console.warn('Usage of deprecated method Cache.prototype.remove');
        this.del(name);
    }

    /**
     * @deprecated
     */
    reset() {
        console.warn('Usage of deprecated method Cache.prototype.reset');
        this.clear();
    }
}

const initialCounterValue = 1000;

let counter = initialCounterValue;

const getRandomNamespace = () => `namespace_${counter++}`;

const instances = new Map();

const cacheFactory = ({ namespace = getRandomNamespace(), ttl = -1 } = {}) => {
    if (instances.get(namespace)) {
        return instances.get(namespace);
    }
    const cache = new Cache(new InMemoryStore(), namespace, ttl);

    instances.set(namespace, cache);
    return cache;
};

export const createDummy = () => {
    const namespace = getRandomNamespace();
    const cache = new Cache(new Store(), namespace);

    instances.set(namespace, cache);
    return cache;
};

const resetCache = (namespace = '') => {
    if (!namespace) {
        const values = instances.values();

        let cache = values.next();

        while (!cache.done) {
            cache.value.clear();
            cache = values.next();
        }

        return;
    }

    if (instances.get(namespace)) {
        instances.get(namespace).clear();
        return;
    }

    throw new Error(`Cache with namespace="${namespace}" was not found.`);
};

const deleteCache = (namespace = '') => {
    if (!namespace) {
        instances.clear();
        return;
    }

    if (instances.get(namespace)) {
        instances.delete(namespace);
        return;
    }

    throw new Error(`Cache with namespace="${namespace}" was not found.`);
};

export { cacheFactory, resetCache, deleteCache };
