import qs from 'qs';
import { cacheFactory } from '../Cache';
import { AbstractREST } from './AbstractREST';
import { createProxyServerError } from '../errors/ServerError';

const DEFAULT_TTL = 5000;

export class ProxyREST extends AbstractREST {
    constructor({ baseUrl, proxyFunction, proxyLinkFunction, namespace = 'ajax', ttl = DEFAULT_TTL }) {
        super({ baseUrl });
        this._proxyFunction = proxyFunction;
        this._proxyLinkFunction = proxyLinkFunction;
        this._baseUrl = baseUrl;
        this._cache = cacheFactory({ namespace, ttl });
    }

    _fetch(url, options) {
        return Promise.resolve(
            this._proxyFunction(this.createUrl(url), {
                type: 'GET',
                contentType: 'application/json',
                data: JSON.stringify(options.body),
                dataType: 'json',
                ...options,
            }),
        )
            .then((data) => {
                this._cache.save(url, data);
                return data;
            })
            .catch((response) => {
                throw createProxyServerError(response);
            });
    }

    _parseUrl(fullUrl) {
        const [, url, query] = fullUrl.match(/([^?]*)\??(.*)/);
        const queryParams = query ? qs.parse(query) : {};

        return { url, queryParams };
    }

    createProxiedUrl(url) {
        return this._proxyLinkFunction(`${this._baseUrl}/${url}`);
    }

    // eslint-disable-next-line no-unused-vars
    async create(url, data) {
        console.warn('ProxyREST.create is not implemented');
    }

    async read(urlWithParams, options = {}) {
        const { url, queryParams } = this._parseUrl(urlWithParams);

        return this._cachedFetch(url, {
            ...options,
            data: queryParams,
        });
    }

    // eslint-disable-next-line no-unused-vars
    async update(url, data) {
        console.warn('ProxyREST.update is not implemented');
    }

    // eslint-disable-next-line no-unused-vars
    async delete(url) {
        console.warn('ProxyREST.delete is not implemented');
    }

    async request({ url, method, body }) {
        return this._fetch(url, {
            type: method,
            body,
        });
    }
}
