import { stringify } from 'qs';
import { resetCache } from '../infrastructure/Cache';
import { serviceLocatorFactory } from '../infrastructure/serviceLocator';
import { ProxyREST } from '../infrastructure/transport/ProxyREST';
import { urlGenerator } from '../utils/proxy';
import { i18nextInit } from '../utils/translate';

// eslint-disable-next-line no-undef
export default class DriveSBSTaskSuite extends TolokaTaskSuite {
    constructor(options) {
        super(options);

        this.locale = ['ru', 'ua', 'by', 'kz', 'uz'].includes(this.getWorkspaceOptions().language) ? 'ru' : 'en';

        try {
            resetCache();
        } catch (e) {
            console.warn('Reset cache error!', e);
        }

        const baseTransportParams = {
            proxyFunction: this.proxy.bind(this),
            proxyLinkFunction: this.proxyLink.bind(this),
        };

        const { media, data } = this.getProxyNames();

        serviceLocatorFactory(
            {
                ProxyREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: media,
                }),
                DataREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: data,
                }),
            },
            { replace: true },
        );
    }

    render() {
        i18nextInit({
            lng: this.locale,
            userKeys: this.getUserLocale(),
        }).then(() => {
            super.render();
        });

        return this;
    }

    getDOMElement() {
        const taskSuiteElement = super.getDOMElement();

        taskSuiteElement.classList.add('drive-sbs');

        return taskSuiteElement;
    }

    /**
     * Get rid of hotkeys, since they mess up React modals
     */
    initHotkeys() {
        this.hotkey.destroy();
        // we need the default submit on Enter here
        // this.getAssignment().hotkey.destroy();
        this.getTasks().forEach((task) => task.hotkey.destroy());
        this.focusTask(0);
    }

    /**
     * Provide plain proxy link for static fetching requests
     * @param {string} path
     * @returns {string}
     */
    proxyLink(path) {
        return urlGenerator(this.getWorkspaceOptions().origin, path, { assignmentId: this.getAssignment().getId() });
    }

    /**
     * Own proxy method implementation, since toloka-handlebars-templates is not imported
     * @param {string} path
     * @param {object} options
     * @param {object} payload
     * @returns {Promise}
     */
    proxy(path, options, payload = {}) {
        const params = {
            ...options,
            path: path + stringify(payload, { addQueryPrefix: true }),
            url: urlGenerator(this.getWorkspaceOptions().origin, path),
        };

        return new Promise((resolve, reject) =>
            this.getAssignment()
                .getSandboxChannel()
                .request('proxy', params)
                .then((response) => {
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        reject({
                            status: 500,
                            message: response.requestId,
                        });
                    }
                })
                .catch((error) => reject(error.data)),
        );
    }

    /**
     * Getter for user-specified locale
     * @returns {{en: {}, ru: {}}}
     */
    getUserLocale() {
        return {
            ru: {},
            en: {},
        };
    }

    getFacts() {
        return {};
    }

    getFactsOnRegistration() {
        return {};
    }

    /**
     * Getter for user proxy names
     * @returns {{data: string, media: string}}
     */
    getProxyNames() {
        return {
            data: 'robot-yang-cars-registration-data',
            media: 'robot-yang-cars-registration',
        };
    }

    /**
     * @param {array} solutions
     * @returns {Promise<*>}
     */
    async validate(solutions) {
        let isValid = true;
        const suiteErrors = [];

        for (const solution of solutions) {
            const taskErrors = await this.getTasksIndexed()[solution.task_id].validate(solution);

            suiteErrors.push(taskErrors);

            if (taskErrors) {
                isValid = false;
            }
        }

        if (isValid) {
            return null;
        } else {
            this.onValidationFail(suiteErrors);
            return suiteErrors;
        }
    }
}
