export default class Storage {
    /**
     * @return {string}
     */
    get EXPIRATIONS_KEY() {
        return 'DriveSBS_expirations';
    }

    constructor() {
        if (!Storage.isSupported()) {
            this._storage = {
                setItem: () => {},
                getItem: () => {},
                removeItem: () => {},
            };

            console.info('Local storage is not supported or user blocks third party cookies and storage');
        } else {
            this._storage = window.localStorage;
        }

        this.clearExpired();
    }

    /**
     * Saves item in persistent client storage (localStorage).
     *
     * @param {string} key
     * @param {*} value
     * @param {Date|number} [expiration] default is 24h
     */
    setItem(key, value, expiration) {
        try {
            const expirations = JSON.parse(this._storage.getItem(this.EXPIRATIONS_KEY) || '{}');

            // eslint-disable-next-line no-magic-numbers
            expirations[key] = expiration ? new Date(expiration).getTime() : new Date().getTime() + 24 * 60 * 60 * 1000;

            this._storage.setItem(this.EXPIRATIONS_KEY, JSON.stringify(expirations));
            this._storage.setItem(key, typeof value === 'string' ? value : JSON.stringify(value));
        } finally {
            // null
        }
    }

    /**
     * Returns item from persistent client storage (localStorage)
     *
     * @param key
     * @returns {*}
     */
    getItem(key) {
        const value = this._storage.getItem(key);

        try {
            return JSON.parse(this._storage.getItem(key));
        } catch (e) {
            return value;
        }
    }

    /**
     * Removes item from persistent client storage (localStorage)
     *
     * @param {string} key
     */
    removeItem(key) {
        this._storage.removeItem(key);
    }

    /**
     * Clears expired items.
     */
    clearExpired() {
        try {
            const expirations = JSON.parse(this._storage.getItem(this.EXPIRATIONS_KEY) || '{}'),
                now = new Date().getTime();

            for (const key in expirations) {
                if (now > expirations[key]) {
                    this.removeItem(key);
                    delete expirations[key];
                }
            }

            this._storage.setItem(this.EXPIRATIONS_KEY, JSON.stringify(expirations));
        } finally {
            // null
        }
    }

    /**
     * Clears all items.
     */
    clear() {
        try {
            const expirations = JSON.parse(this._storage.getItem(this.EXPIRATIONS_KEY) || '{}');

            for (const key in expirations) {
                this.removeItem(key);
            }

            this.removeItem(this.EXPIRATIONS_KEY);
        } finally {
            // null
        }
    }

    /**
     * Clears expired items. Just for consistent interface.
     */
    destroy() {
        this.clearExpired();
    }

    /**
     * Checks if storage is supported
     *
     * @return {boolean}
     */
    static isSupported() {
        try {
            // eslint-disable-next-line no-magic-numbers
            const STORAGE_TEST_KEY = `DriveSBS_ls_test_${(Math.random() * 0xffffff) | 0}`;
            const storage = window.localStorage;

            storage.setItem(STORAGE_TEST_KEY, 'test_value');
            storage.removeItem(STORAGE_TEST_KEY);
        } catch (error) {
            return false;
        }

        return true;
    }
}
