/* eslint-disable camelcase */
import React from 'react';
import PropTypes from 'prop-types';
import { Form, Input, Checkbox, Radio } from 'antd';
import { DOCUMENT_TYPES, VERDICT_OK, VERDICT_OTHER } from '../../domain/documents';
import { serviceLocator } from '../../infrastructure/serviceLocator';
import UsecaseRunError from '../../infrastructure/errors/UsecaseRunError';
import { translate } from '../../utils/translate';
import { withHotkey } from '../Utility/WithHotkey';
import { SidebarHeader } from '../SidebarHeader';
import { Criteria } from '../Criteria';
import './Sidebar.less';

const formLayout = {
    wrapperCol: { span: 24 },
};

export class SidebarView extends React.PureComponent {
    static propTypes = {
        onSubmit: PropTypes.func.isRequired,
        resetTaskError: PropTypes.func.isRequired,
        taskError: PropTypes.string,
        videoPlayed: PropTypes.bool,
        videoError: PropTypes.bool,
        facts: PropTypes.object,
        verdicts: PropTypes.arrayOf(PropTypes.string),
        document: PropTypes.oneOf(DOCUMENT_TYPES),
        scaleOptions: PropTypes.shape({ multipleSelect: PropTypes.bool, otherWithComment: PropTypes.bool }),
    };

    formRef = React.createRef();
    sidebarRef = React.createRef();
    commentRef = React.createRef();

    handleFieldsChange = (values) => {
        const { resetTaskError, facts, document, verdicts, scaleOptions, onSubmit } = this.props;
        const fields = Object.keys(facts);

        resetTaskError();

        if (!values[0]) {
            return;
        }

        const { name, value } = values[0];

        if (name[0] === VERDICT_OK && value) {
            this.formRef.current.setFieldsValue({
                ...fields.reduce((results, field) => ({ ...results, [field]: false }), {}),
                [VERDICT_OTHER]: false,
            });

            return onSubmit({ verdicts: { [document]: [VERDICT_OK] }, comments: { [document]: null } });
        }

        if (fields.includes(name[0]) || name[0] === VERDICT_OTHER) {
            this.formRef.current.setFieldsValue({ [VERDICT_OK]: false });

            if (!scaleOptions.multipleSelect) {
                this.formRef.current.setFieldsValue({
                    ...fields.reduce((results, field) => ({ ...results, [field]: false }), {}),
                    [VERDICT_OTHER]: false,
                    [name[0]]: value,
                });
            }

            let results = verdicts.filter((verdict) => verdict !== VERDICT_OK && verdict !== name[0]);

            results = value ? (scaleOptions.multipleSelect ? [...results, name[0]] : [name[0]]) : results;

            return onSubmit({ verdicts: { [document]: results } });
        }

        if (name[0] === 'description') {
            onSubmit({ comments: { [document]: value } });
        }
    };

    handleCommentKeyDown = (e) => {
        const hasModifier = e.ctrlKey || e.metaKey;

        if (e.key !== 'Enter') {
            return;
        }

        if (!hasModifier) {
            return e.preventDefault();
        }

        this.commentRef.current?.blur();
    };

    validateAll = async () => {
        try {
            await this.formRef.current.validateFields();
        } catch {
            // foo
        }
    };

    componentDidMount() {
        window.requestAnimationFrame(() => {
            this.sidebarRef.current.focus();
        });

        serviceLocator.validateAndCollect = async () => {
            try {
                await this.formRef.current.validateFields();
            } catch {
                throw new UsecaseRunError();
            }
        };
    }

    componentWillUnmount() {
        serviceLocator.validateAndCollect = null;
    }

    render() {
        const { facts, taskError, videoPlayed, videoError, document, scaleOptions } = this.props;
        const fields = Object.entries(facts).map(([name, label]) => ({
            name,
            label,
        }));

        const criteria = fields.map(({ name, label }, index) => {
            const haveVideoErrorCriteria = fields.some(({ name }) => name === 'VIDEO_ERROR');
            const isDisabledOnError = haveVideoErrorCriteria ? name !== 'VIDEO_ERROR' : false;

            return (
                <Criteria
                    key={name}
                    name={name}
                    label={label}
                    index={index}
                    disabled={videoError ? isDisabledOnError : !videoPlayed}
                    multipleSelect={scaleOptions.multipleSelect}
                />
            );
        });

        const dependencies = [...fields.map(({ name }) => [name]), [VERDICT_OTHER]];

        const InputComponent = scaleOptions.multipleSelect ? withHotkey(Checkbox) : withHotkey(Radio);

        return (
            <div className="sidebar" ref={this.sidebarRef} tabIndex={-1}>
                <SidebarHeader document={document} onBeforeChange={this.validateAll} />

                <Form
                    {...formLayout}
                    ref={this.formRef}
                    onFieldsChange={this.handleFieldsChange}
                    labelAlign="left"
                    hideRequiredMark
                >
                    {criteria}

                    {scaleOptions.otherWithComment && (
                        <>
                            <Form.Item name={VERDICT_OTHER} valuePropName="checked">
                                <InputComponent hotKey={{ code: 'KeyQ', label: 'Q' }} disabled={!videoPlayed}>
                                    {translate('verdict_other')}
                                </InputComponent>
                            </Form.Item>

                            <Form.Item
                                noStyle
                                shouldUpdate={(prevValues, currentValues) =>
                                    prevValues[VERDICT_OTHER] !== currentValues[VERDICT_OTHER]
                                }
                            >
                                {({ getFieldValue }) =>
                                    getFieldValue([VERDICT_OTHER]) && (
                                        <Form.Item
                                            name="description"
                                            preserve={false}
                                            rules={[
                                                ({ getFieldValue }) => ({
                                                    validator(_, value) {
                                                        const isRequired = getFieldValue([VERDICT_OTHER]);

                                                        if (!value && isRequired) {
                                                            return Promise.reject(
                                                                translate('error_validation_description'),
                                                            );
                                                        }

                                                        return Promise.resolve();
                                                    },
                                                }),
                                            ]}
                                        >
                                            <Input
                                                ref={this.commentRef}
                                                autoFocus
                                                className="sidebar__description"
                                                placeholder={translate('common_description_placeholder')}
                                                onKeyDown={this.handleCommentKeyDown}
                                            />
                                        </Form.Item>
                                    )
                                }
                            </Form.Item>
                        </>
                    )}

                    <Form.Item
                        name={VERDICT_OK}
                        dependencies={dependencies}
                        valuePropName="checked"
                        validateStatus={taskError ? 'error' : null}
                        help={taskError}
                        rules={[
                            ({ getFieldsValue }) => ({
                                validator(_, value) {
                                    const formValues = getFieldsValue();

                                    if (!value && Object.keys(formValues).every((field) => !formValues[field])) {
                                        return Promise.reject(translate('error_validation_criteria'));
                                    }
                                    return Promise.resolve();
                                },
                            }),
                        ]}
                    >
                        <InputComponent hotKey={{ code: 'KeyW', label: 'W' }} disabled={!videoPlayed}>
                            {translate('verdict_ok')}
                        </InputComponent>
                    </Form.Item>
                </Form>
            </div>
        );
    }
}
