import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Typography, Button } from 'antd';
import { ArrowLeftOutlined, ArrowRightOutlined } from '@ant-design/icons';
import { translate } from '../../utils/translate';
import { DOCUMENT_TYPES } from '../../domain/documents';
import { Hint } from '../Utility/Hint';
import './SidebarHeader.less';

export class SidebarHeaderView extends PureComponent {
    static propTypes = {
        document: PropTypes.oneOf(DOCUMENT_TYPES).isRequired,
        activeDocumentIndex: PropTypes.number.isRequired,
        setActiveDocumentIndex: PropTypes.func.isRequired,
        availableDocuments: PropTypes.number.isRequired,
        onBeforeChange: PropTypes.func.isRequired,
    };

    /**
     * Switch tasks
     * @param {number} modifier
     * @returns {Promise<void>}
     */
    handleDocumentSwitch = async (modifier) => {
        const { availableDocuments, activeDocumentIndex, setActiveDocumentIndex } = this.props;
        const newIndex = activeDocumentIndex + modifier;

        if (newIndex < 0 || newIndex >= availableDocuments) {
            return;
        }

        const isValidated = await this.props.onBeforeChange();

        if (isValidated) {
            setActiveDocumentIndex(newIndex);
        }
    };

    handleHotKey = (e) => {
        if (
            !['ArrowLeft', 'ArrowRight'].includes(e.code) ||
            (document.activeElement &&
                document.activeElement.tagName === 'INPUT' &&
                document.activeElement.type === 'text')
        ) {
            return;
        }

        this.handleDocumentSwitch(e.code === 'ArrowLeft' ? -1 : 1);

        e.preventDefault();
    };

    componentDidMount() {
        document.addEventListener('keydown', this.handleHotKey);
    }

    componentWillUnmount() {
        document.removeEventListener('keydown', this.handleHotKey);
    }

    render() {
        const { activeDocumentIndex, availableDocuments } = this.props;
        const isPrevDisabled = activeDocumentIndex === 0;
        const isNextDisabled = activeDocumentIndex === availableDocuments - 1;

        return (
            <section className="sidebar__header">
                {availableDocuments > 1 && (
                    <span className="button__previous">
                        <Hint title={translate('common_task_previous')}>
                            <Button
                                className="button__previous"
                                icon={<ArrowLeftOutlined />}
                                disabled={isPrevDisabled}
                                onClick={() => this.handleDocumentSwitch(-1)}
                            />
                        </Hint>
                    </span>
                )}

                <Typography.Title level={4} className="sidebar__title">
                    {translate('common_task_title')}: {translate(this.props.document)}
                </Typography.Title>

                {availableDocuments > 1 && (
                    <span className="button__next">
                        <Hint title={translate('common_task_next')}>
                            <Button
                                icon={<ArrowRightOutlined />}
                                disabled={isNextDisabled}
                                onClick={() => this.handleDocumentSwitch(1)}
                            />
                        </Hint>
                    </span>
                )}
            </section>
        );
    }
}
