/* eslint-disable camelcase */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import Storage from '../../utils/storage';
import { DOCUMENT_TYPES } from '../../domain/documents';
import SplitPane from 'react-split-pane';
import { VideoPlayer } from '../VideoPlayer';
import { Sidebar } from '../Sidebar';
import './TaskPageLayout.less';

// eslint-disable-next-line no-magic-numbers
const EXPIRATION_TIME = 365 * 24 * 60 * 60 * 1000;
const DEFAULT_PANE_SIZE = 515;

export class TaskPageLayoutView extends PureComponent {
    static propTypes = {
        videoUrls: PropTypes.arrayOf(
            PropTypes.shape({
                type: PropTypes.oneOf(DOCUMENT_TYPES),
                src: PropTypes.string.isRequired,
            }),
        ).isRequired,
        activeDocumentIndex: PropTypes.number.isRequired,
        availableDocumentTypes: PropTypes.arrayOf(PropTypes.oneOf(DOCUMENT_TYPES)),
        setVideoError: PropTypes.func.isRequired,
        setVideoPlayed: PropTypes.func.isRequired,
    };

    get splitSize() {
        return this.storage.getItem('splitSizeDV') || DEFAULT_PANE_SIZE;
    }

    constructor(props) {
        super(props);

        this.storage = new Storage();
    }

    componentWillUnmount() {
        this.storage.destroy();
    }

    handleSplitSize = (size) => this.storage.setItem('splitSizeDV', size, new Date().getTime() + EXPIRATION_TIME);

    handleVideoError = (document) => {
        this.props.setVideoError({document, value: true});
        this.props.setVideoPlayed({document, value: false});
    }

    handleVideoPlayed = (document) => {
        this.props.setVideoError({document, value: false});
        this.props.setVideoPlayed({document, value: true});
    }

    render() {
        const { videoUrls, activeDocumentIndex, availableDocumentTypes } = this.props;
        const src = videoUrls[activeDocumentIndex]?.src;
        const currentDocument = availableDocumentTypes[activeDocumentIndex];

        return (
            <div className="task-pane">
                <div className="task-pane__row">
                    <SplitPane
                        primary="second"
                        split="vertical"
                        minSize={DEFAULT_PANE_SIZE}
                        defaultSize={this.splitSize}
                        onDragFinished={this.handleSplitSize}
                    >
                        <main className="task-pane__main">
                            <VideoPlayer
                                src={src}
                                onEnded={() => this.handleVideoPlayed(currentDocument)}
                                onError={() => this.handleVideoError(currentDocument)}
                            />
                        </main>

                        <aside className="task-pane__aside">
                            <Sidebar document={currentDocument} />
                        </aside>
                    </SplitPane>
                </div>
            </div>
        );
    }
}
