/* eslint-disable no-undefined */
import React from 'react';
import propType from 'prop-types';
import { translate } from '../../utils/translate';
import { Spin } from 'antd';
import { LoadingOutlined } from '@ant-design/icons';

export class Async extends React.Component {
    static propTypes = {
        promise: propType.shape({
            then: propType.func.isRequired,
            catch: propType.func.isRequired,
        }).isRequired,
        renderLoadingLayout: propType.func,
        renderErrorLayout: propType.func,
    };

    static defaultProps = {
        renderLoadingLayout: () => <Spin indicator={<LoadingOutlined style={{ fontSize: 24 }} />} />,
        renderErrorLayout: (error) => <div className="error">{error.message}</div>,
    };

    static getDerivedStateFromProps(props, state) {
        if (state.promise === props.promise) return null;

        return {
            ready: false,
            promise: props.promise,
            error: undefined,
            value: undefined,
        };
    }

    mounted = false;

    constructor(props) {
        super(props);

        this.state = {
            ready: false,
            promise: props.promise,
        };
    }

    componentDidMount() {
        this.mounted = true;
        this.wait();
    }

    componentWillUnmount() {
        this.mounted = false;
    }

    componentDidUpdate() {
        if (!this.state.ready) {
            this.wait();
        }
    }

    render() {
        const { renderLoadingLayout, renderErrorLayout } = this.props;

        if (this.state.ready) {
            const { value, error } = this.state;

            if (typeof error !== 'undefined') {
                return renderErrorLayout(error);
            }

            return this.props.children(value);
        }

        return renderLoadingLayout();
    }

    async wait() {
        const promise = this.state.promise;

        try {
            const data = await promise;

            if (this.mounted) {
                this.setState({
                    value: data,
                    ready: true,
                    promise,
                });
            }
        } catch (error) {
            const message = error && error.message ? error.message : translate('error_common');

            if (this.mounted) {
                this.setState({
                    ready: true,
                    error: {
                        message,
                        status: error.status,
                    },
                    promise,
                });
            }
        }
    }
}
