import { parse, stringify } from 'qs';
import { cacheFactory } from '../Cache';
import { AbstractREST } from './AbstractREST';
import { createProxyServerError } from '../errors/ServerError';

const DEFAULT_TTL = 5000;

export class ProxyREST extends AbstractREST {
    constructor({ baseUrl, sandboxChannel, assignmentId, origin, namespace = 'ajax', ttl = DEFAULT_TTL }) {
        super({ baseUrl });
        this._baseUrl = baseUrl;
        this._sandboxChannel = sandboxChannel;
        this._assignmentId = assignmentId;
        this._origin = origin;
        this._cache = cacheFactory({ namespace, ttl });
    }

    _proxyRequest(path, options) {
        // в свежих версиях фронта Толоки url формируется автоматически
        // в этом месте мы формируем его только для отладки в TTS <= 1.1.34, так как там остался старый механизм
        const url =
            window?.location?.host === 'localhost:9000'
                ? this.getProxyRequestUrl(path.replace(this._baseUrl, ''))
                : null;

        return new Promise((resolve, reject) =>
            this._sandboxChannel
                .request('proxy', { ...options, path, url })
                .then((response) => {
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        reject({
                            status: 500,
                            message: response.requestId,
                        });
                    }
                })
                .catch((error) => reject(error.data)),
        );
    }

    _fetch(url, options) {
        return Promise.resolve(
            this._proxyRequest(this.createUrl(url), {
                type: 'GET',
                contentType: 'application/json',
                data: options.body ? JSON.stringify(options.body) : options.data,
                dataType: 'json',
                ...options,
            }),
        )
            .then((data) => {
                this._cache.save(url, data);
                return data;
            })
            .catch((response) => {
                throw createProxyServerError(response);
            });
    }

    _parseUrl(fullUrl) {
        const [, url, query] = fullUrl.match(/([^?]*)\??(.*)/);
        const queryParams = query ? parse(query) : {};

        return { url, queryParams };
    }

    getProxyRequestUrl(path, data = {}) {
        const origin = this._origin;
        const urlPath = path.replace(/^\/?/, '');

        return (
            [origin, 'api/proxy', `${this._baseUrl}/${urlPath}`].join('/') +
            stringify({ ...data, assignmentId: this._assignmentId }, { addQueryPrefix: true })
        );
    }

    // eslint-disable-next-line no-unused-vars
    async create(url, data) {
        console.warn('ProxyREST.create is not implemented');
    }

    async read(urlWithParams, options = {}) {
        const { url, queryParams } = this._parseUrl(urlWithParams);

        return this._cachedFetch(url, {
            ...options,
            data: queryParams,
        });
    }

    // eslint-disable-next-line no-unused-vars
    async update(url, data) {
        console.warn('ProxyREST.update is not implemented');
    }

    // eslint-disable-next-line no-unused-vars
    async delete(url) {
        console.warn('ProxyREST.delete is not implemented');
    }

    async request({ url, method, body }) {
        return this._fetch(url, {
            type: method,
            body,
        });
    }
}
