import {store} from '../domain/store';
import {serviceLocator} from '../infrastructure/serviceLocator';
import {stringify} from "qs";

export default class DriveAssignment extends window.TolokaAssignment {
    /**
     * Provides solutions to the app if they are valid.
     *
     * @param {Function} strategy
     * @param {Function} errorCallback - called if validation errors
     */
    provideSolutions(
        strategy = function(solutions) {
            this.getSandboxChannel().triggerOut('assignment:submit', { solutions, assignmentId: this.getId() });
        },
        // eslint-disable-next-line no-unused-vars
        errorCallback = function() {},
    ) {
        const solutions = this.getTaskSuite().getSolutions();

        const {task_type: taskType} = store.getState()?.input;

        Promise.resolve(this.getTaskSuite().validate(solutions)).then(async (errors) => {
            if (!errors) {
                if (taskType === 'reissue_license') {
                    this.setStatus(strategy, solutions);
                } else if (taskType === 'registration_on_chat') {
                    await this.addToQueueRequest(strategy, solutions);
                } else {
                    strategy.call(this, solutions);
                }
            } else {
                this.getSandboxChannel().triggerOut('assignment:validation:fail', errors);
            }
        });
    }

    setStatus(strategy, solutions) {
        const {user_id: userId, secret: secretId} = store.getState()?.input;

        const verdict = Object.entries(store.getState()?.solution?.verdicts ?? {})?.[0];
        const statusTypes = this.getTaskSuite().getSetStatusType();
        const finalVerdict = [statusTypes[verdict[0]], verdict[1][0]];

        this.setStatusRequest(userId, finalVerdict, secretId)
            .then(() => {
                // eslint-disable-next-line camelcase
                solutions[0].output_values.send_server_status = 'ok';
                strategy.call(this, solutions);
            })
            .catch((error) => {
                const SUCCESS_STATUS = 200;

                if (error?.status && error?.status !== SUCCESS_STATUS) {
                    console.log(error.status);
                    solutions[0].output_values.send_server_status = 'error';
                    strategy.call(this, solutions);
                } else {
                    solutions[0].output_values.send_server_status = 'ok';
                    strategy.call(this, solutions);
                }
            });
    }

    async setStatusRequest(userId, verdict, secretId) {
        const queryParams = stringify({ secretId }, { addQueryPrefix: true });

        await serviceLocator.ChecksREST.request({
            url:`${queryParams}`,
            method: 'POST',
            body: {
                // eslint-disable-next-line camelcase
                user_id: userId,
                type: verdict[0],
                status: verdict[1] ?? 'can not get status',
            }
        });
    }

    async addToQueueRequest(strategy, solutions) {
        const { secret } = store.getState().input;
        const queryParams = stringify({ secretId: secret }, { addQueryPrefix: true });
        const statusTypes = this.getTaskSuite().getSetStatusType();

        const verdicts = solutions[0].output_values.verdicts;
        const other = solutions[0].output_values.other;
        const output_values = {};
        output_values.verdicts = Object.keys(verdicts)
            .reduce((prev, curr) => {
                prev[statusTypes[curr]] = verdicts[curr];
                return prev;
            }, {});

        solutions[0].output_values = { ...output_values, other };

        await serviceLocator.DataREST.request({
            url: `assignment/queue${queryParams}`,
            method: 'POST',
        }).then(() => {
            solutions[0].output_values.send_server_status = 'ok';
            strategy.call(this, solutions);
        }).catch((error) => {
            const SUCCESS_STATUS = 200;

            if (error?.status && error?.status !== SUCCESS_STATUS) {
                solutions[0].output_values.send_server_status = 'error';
                strategy.call(this, solutions);
            } else {
                solutions[0].output_values.send_server_status = 'ok';
                strategy.call(this, solutions);
            }
        });
    }
}
