import { resetCache } from '../infrastructure/Cache';
import { serviceLocatorFactory } from '../infrastructure/serviceLocator';
import { ProxyREST } from '../infrastructure/transport/ProxyREST';
import { i18nextInit } from '../utils/translate';

// eslint-disable-next-line no-undef
export default class DriveTaskSuite extends TolokaTaskSuite {
    constructor(options) {
        super(options);

        this.locale = ['ru', 'ua', 'by', 'kz', 'uz'].includes(this.getWorkspaceOptions().language) ? 'ru' : 'en';

        try {
            resetCache();
        } catch (e) {
            console.warn('Reset cache error!', e);
        }

        const baseTransportParams = {
            sandboxChannel: this.getAssignment().getSandboxChannel(),
            assignmentId: this.getAssignment().getId(),
            origin: this.getWorkspaceOptions().origin,
        };

        const { checks, media, data } = this.getProxyNames();

        serviceLocatorFactory(
            {
                ProxyREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: media,
                }),
                ChecksREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: checks,
                }),
                DataREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: data,
                }),
            },
            { replace: true },
        );
    }

    render() {
        i18nextInit({
            lng: this.locale,
            userKeys: this.getUserLocale(),
        }).then(() => {
            super.render();
        });

        return this;
    }

    getDOMElement() {
        const taskSuiteElement = super.getDOMElement();

        taskSuiteElement.classList.add('drive-sbs');

        return taskSuiteElement;
    }

    /**
     * Get rid of hotkeys, since they mess up React modals
     */
    initHotkeys() {
        this.hotkey.destroy();
        // this.getAssignment().hotkey.destroy();
        this.getTasks().forEach((task) => task.hotkey.destroy());
        this.focusTask(0);
    }

    /**
     * Getter for user-specified locale
     * @returns {{en: {}, ru: {}}}
     */
    getUserLocale() {
        return {
            ru: {},
            en: {},
        };
    }

    getFacts() {
        return {};
    }

    /**
     * Getter for user-specified proxy names
     * @returns {{checks: string, media: string}}
     */
    getProxyNames() {
        return {
            checks: 'robot-yang-cars-registration-documents-checks/set_status',
            data: 'robot-yang-cars-registration-data',
            media: 'robot-yang-cars-registration',
        };
    }

    /**
     * Getter for misc scale options
     * @returns {{multipleSelect: boolean, otherWithComment: boolean}}
     */
    getScaleOptions() {
        return {
            multipleSelect: false,
            otherWithComment: false,
        };
    }

    /**
     * @param {array} solutions
     * @returns {Promise<*>}
     */
    async validate(solutions) {
        let isValid = true;
        const suiteErrors = [];

        for (const solution of solutions) {
            const taskErrors = await this.getTasksIndexed()[solution.task_id].validate(solution);

            suiteErrors.push(taskErrors);

            if (taskErrors) {
                isValid = false;
            }
        }

        if (isValid) {
            return null;
        } else {
            this.onValidationFail(suiteErrors);
            return suiteErrors;
        }
    }
}
