#include "crypto.h"

#include <contrib/libs/openssl/include/openssl/evp.h>

#include <library/cpp/openssl/holders/evp.h>


TCryptoGCMProcessor::TCryptoGCMProcessor(const EMode mode, const unsigned char* encryptionKey)
    : Mode(mode)
{
    ui16 encryptionKeySize = 0;
    switch (Mode) {
        case EMode::GCM_128:
            encryptionKeySize = 16;
            break;
        case EMode::GCM_256:
            encryptionKeySize = 32;
            break;
    }
    for (size_t i = 0; i < encryptionKeySize; ++i) {
        EncryptionKey.push_back(encryptionKey[i]);
    }
}

TCryptoGCMProcessor::TCryptoGCMProcessor(const EMode mode, const TVector<unsigned char>& encryptionKey)
    : Mode(mode)
    , EncryptionKey(encryptionKey)
{
    switch (Mode) {
        case EMode::GCM_128:
            Y_ASSERT(EncryptionKey.size() == 16);
            break;
        case EMode::GCM_256:
            Y_ASSERT(EncryptionKey.size() == 32);
            break;
    }
}

const EVP_CIPHER* GetEnvCipher(TCryptoGCMProcessor::EMode mode) {
    switch (mode) {
        case TCryptoGCMProcessor::EMode::GCM_128:
            return EVP_aes_128_gcm();
        case TCryptoGCMProcessor::EMode::GCM_256:
            return EVP_aes_256_gcm();
    }
}

TMaybe<TString> TCryptoGCMProcessor::Decrypt(const TString& encryptedContent, const TString& iv, const TString& tag) const {
    auto ctx = EVP_CIPHER_CTX_new();

    if(!EVP_DecryptInit(ctx, GetEnvCipher(Mode), nullptr, nullptr))  {
        ERROR_LOG << "EVP_DecryptInit failed (the very beginning of decryption)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if(!EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_IVLEN, iv.size(), nullptr)) {
        ERROR_LOG << "EVP_CIPHER_CTX_ctrl failed on EVP_CTRL_GCM_SET_IVLEN (setting custom length of init vector)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_DecryptInit(ctx, nullptr, EncryptionKey.data(), (unsigned char*)iv.data())) {
        ERROR_LOG << "EVP_DecryptInit failed (feeding actual key and iv)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_TAG, tag.size(), (void*)tag.data())) {
        ERROR_LOG << "EVP_CIPHER_CTX_ctrl on EVP_CTRL_GCM_SET_TAG (tag setup)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    TString result = TString(encryptedContent.size() + EVP_MAX_BLOCK_LENGTH, 0);
    int plainLen = 0;
    int tailLen = 0;

    if (!EVP_DecryptUpdate(ctx, (unsigned char*)result.data(), &plainLen, (unsigned char*)encryptedContent.data(), encryptedContent.size())) {
        ERROR_LOG << "EVP_DecryptUpdate failed  (doing actual decryption)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_DecryptFinal(ctx, nullptr, &tailLen)) {
        ERROR_LOG << "EVP_DecryptFinal failed (finalization)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    result.resize(plainLen + tailLen);
    EVP_CIPHER_CTX_free(ctx);

    return result;
}

TMaybe<TCryptoGCMProcessor::Encrypted> TCryptoGCMProcessor::Encrypt(const TString& content, const TString& iv) const {
    auto ctx = EVP_CIPHER_CTX_new();

    if (!EVP_EncryptInit(ctx, GetEnvCipher(Mode), nullptr, nullptr)) {
        ERROR_LOG << "EVP_EncryptInit failed (the very beginning of encryption)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if(!EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_IVLEN, iv.size(), nullptr)) {
        ERROR_LOG << "EVP_CIPHER_CTX_ctrl failed on EVP_CTRL_GCM_SET_IVLEN (setting custom length of init vector)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_EncryptInit(ctx, nullptr, EncryptionKey.data(), (unsigned char*)iv.data())) {
        ERROR_LOG << "EVP_EncryptInit failed (feeding actual key and iv)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    TString result = TString(content.size() + EVP_MAX_BLOCK_LENGTH, 0);
    TString tag = TString(16, 0);
    int plainLen = 0;
    int tailLen = 0;

    if (!EVP_EncryptUpdate(ctx, (unsigned char*)result.data(), &plainLen, (unsigned char*)content.data(), content.size())) {
        ERROR_LOG << "EVP_EncryptUpdate failed (doing actual encryption)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_EncryptFinal(ctx, (unsigned char*)tag.data(), &tailLen)) {
        ERROR_LOG << "EVP_EncryptFinal failed (finalization)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    if (!EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_GET_TAG, tag.size(), (void*)tag.data())) {
        ERROR_LOG << "EVP_CIPHER_CTX_ctrl failed with EVP_CTRL_GCM_GET_TAG (tag acquisition)" << Endl;
        EVP_CIPHER_CTX_free(ctx);
        return {};
    }

    result.resize(tailLen + plainLen);
    EVP_CIPHER_CTX_free(ctx);

    return TCryptoGCMProcessor::Encrypted(tag, result);
}

ui8 TCryptoGCMProcessor::HexDigitToInt(const char digit) {
    if (digit >= '0' && digit <= '9') {
        return digit - '0';
    }
    return ToLower(digit) - 'a' + 10;
}
