#include "client.h"

#include <drive/library/cpp/openssl/rsa.h>

#include <util/charset/wide.h>
#include <util/datetime/base.h>
#include <util/string/builder.h>
#include <util/string/hex.h>
#include <util/string/join.h>

namespace NDrive::NAutocode {
    // TAutocodeAuthProvider

    void TAutocodeAuthProvider::AuthorizeData(NJson::TJsonValue& data) const {
        data.InsertValue("SecurityInfoData", GetSecurityData());
    }

    NJson::TJsonValue TAutocodeAuthProvider::GetSecurityData(const TString& seed) const {
        const TInstant now = TInstant::Now();

        const TString nonceSeed = (!!seed) ? seed : ::ToString(now.MicroSeconds());
        TString nonce = HexEncode(NOpenssl::CalcSHA1(seed));
        nonce.to_upper();

        TStringStream ss;
        ss << AuthConfig.GetSecret() << "&" << AuthConfig.GetServiceId() << "&" << nonce << "&" << now.Seconds();

        const TString token = NOpenssl::CalcHMACToken(AuthConfig.GetSecret(), ss.Str());

        NJson::TJsonValue data;
        data.InsertValue("Nonce", nonce);
        data.InsertValue("SystemId", AuthConfig.GetServiceId());
        data.InsertValue("Timestamp", now.Seconds());
        data.InsertValue("Token", token);
        return data;
    }

    // TAutocodeEvacuationClient

    TAutocodeEvacuationClient::TAutocodeEvacuationClient(const TAutocodeClientConfig& config, const TString& apiName)
        : TBase(*config.GetEvacuationApiConfig(), apiName, new TAutocodeAuthProvider(config))
    {
    }

    bool TAutocodeEvacuationClient::CheckIsCarEvacuated(const TString& carNumber, bool& isEvacuated, TMessagesCollector& errors) const {
        TEvacuationInfo evacuationInfo;
        if (!GetCarEvacuationStatus(carNumber, evacuationInfo, errors)) {
            return false;
        }
        isEvacuated = evacuationInfo.IsEvacuated();
        return true;
    }

    bool TAutocodeEvacuationClient::GetCarEvacuationStatus(const TString& carNumber, TEvacuationInfo& evacuationInfo, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::CheckCarEvacuationStatus, CreateCarEvacuationCheckRequest(carNumber), rawResult, errors)) {
            return false;
        }
        if (!rawResult.Has("status") || rawResult["status"].GetString() != "Success") {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error checking car evacuation status (car number - " << carNumber << "): " << rawResult["message"].GetStringRobust());
            return false;
        }
        return evacuationInfo.DeserializeFromJson(rawResult["response"]);
    }

    NNeh::THttpRequest TAutocodeEvacuationClient::CreateCarEvacuationCheckRequest(const TString& carNumber) const {
        NJson::TJsonValue payload(NJson::JSON_MAP);

        TUtf16String wideNumber = TUtf16String::FromUtf8(carNumber);
        wideNumber.to_upper();
        payload.InsertValue("RegistrationPlate", WideToUTF8(wideNumber));

        return CreateCommonRequest(Config.GetUri(), ERequestMethod::POST, std::move(payload), ERequestContentType::Json);
    }

    // TAutocodeDriverLicenseClient

    TAutocodeDriverLicenseClient::TAutocodeDriverLicenseClient(const TAutocodeClientConfig& config, const TString& apiName)
        : TBase(*config.GetDriverLicenseApiConfig(), apiName, new TAutocodeAuthProvider(config))
    {
    }

    bool TAutocodeDriverLicenseClient::GetLicenseInfo(const TString& licenseNumber, const TInstant issueDate, TDriverLicenseInfo& licenseInfo, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::GetDriverLicenseInfo, CreateGetLicenseInfoRequest(licenseNumber, issueDate), rawResult, errors)) {
            return false;
        }

        if (!rawResult.Has("status") || rawResult["status"].GetString() != "Success") {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error collecting driver license info (number - " << licenseNumber << "): " << rawResult["message"].GetStringRobust());
            return false;
        }

        return licenseInfo.DeserializeFromJson(rawResult["response"]);
    }

    NNeh::THttpRequest TAutocodeDriverLicenseClient::CreateGetLicenseInfoRequest(const TString& licenseNumber, const TInstant issueDate) const {
        NJson::TJsonValue payload(NJson::JSON_MAP);
        payload["Number"] = licenseNumber;
        payload["IssueDate"] = issueDate.FormatGmTime("%d.%m.%Y");  // GMT time
        return CreateCommonRequest(Config.GetInfoUri(), ERequestMethod::POST, std::move(payload), ERequestContentType::Json);
    }

    // TAutocodeFinesClient

    TAutocodeFinesClient::TAutocodeFinesClient(const TAutocodeClientConfig& config, const TString& apiName)
        : TBase(*config.GetFinesApiConfig(), apiName, new TAutocodeAuthProvider(config))
    {
    }

    bool TAutocodeFinesClient::ProcessModifySubscribersResponse(const NJson::TJsonValue& rawResult, TMessagesCollector& errors) const {
        if (!rawResult.Has("statusCode") || rawResult["statusCode"].GetString() != "Success") {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error modifying subscribers: " << rawResult["message"].GetStringRobust());
            return false;
        }
        return true;
    }

    bool TAutocodeFinesClient::AddSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::AddSubscribers, CreateAddSubscribersRequest(documentType, documentNumbers), rawResult, errors)) {
            return false;
        }
        return ProcessModifySubscribersResponse(rawResult, errors);
    }

    bool TAutocodeFinesClient::RemoveSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::RemoveSubscribers, CreateRemoveSubscribersRequest(documentType, documentNumbers), rawResult, errors)) {
            return false;
        }
        return ProcessModifySubscribersResponse(rawResult, errors);
    }

    bool TAutocodeFinesClient::ConfirmFines(const TVector<TString>& externalFineIds, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::ConfirmFines, CreateFineConfirmationRequest(externalFineIds), rawResult, errors)) {
            return false;
        }
        return true;
    }

    bool TAutocodeFinesClient::GetFines(TVector<TAutocodeFine>& fines, TMessagesCollector& errors, const EPenaltyCheckPolicy policy, ui64 limit, ui64 offset) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::GetFines, CreateFineCollectingRequest(policy, limit, offset), rawResult, errors)) {
            return false;
        }
        if (!rawResult.IsArray()) {
            errors.AddMessage(__LOCATION__, "Incorrect json format");
            return false;
        }

        for (const auto &rawFine : rawResult.GetArray()) {
            TAutocodeFine fine;
            if (!fine.DeserializeFromJson(rawFine)) {
                return false;
            }
            fines.push_back(std::move(fine));
        }

        return true;
    }

    bool TAutocodeFinesClient::GetViolationPhotos(const TString& rulingNumber, TVector<TAutocodeFinePhoto>& photos, TMessagesCollector& errors) const {
        NJson::TJsonValue rawResult;
        if (!SendRequest(EAutocodeOperationType::GetViolationPhotos, CreateViolationPhotoCollectingRequest(rulingNumber), rawResult, errors)) {
            return false;
        }

        if(!rawResult.Has("photos") || !rawResult["photos"].IsArray()) {
            errors.AddMessage(__LOCATION__, "Incorrect json format");
            return false;
        }

        for (const auto& rawPhoto : rawResult["photos"].GetArray()) {
            TAutocodeFinePhoto photo;
            if (!photo.DeserializeFromJson(rawPhoto)) {
                return false;
            }
            photos.push_back(std::move(photo));
        }

        return true;
    }

    NJson::TJsonValue TAutocodeFinesClient::CreateModifySubscribersRequestData(const EDocumentType documentType, const TVector<TString>& documentNumbers) const {
        NJson::TJsonValue payload(NJson::JSON_MAP);

        const TString documentTypeStr = ToString(documentType);

        auto& subscribers = payload.InsertValue("subscribers", NJson::JSON_ARRAY);
        for (const auto& documentNumber : documentNumbers) {
            auto& subscriber = subscribers.AppendValue(NJson::JSON_MAP);
            subscriber["DocumentType"] = documentTypeStr;
            subscriber["DocumentNumber"] = documentNumber;
        }

        return payload;
    }

    NNeh::THttpRequest TAutocodeFinesClient::CreateAddSubscribersRequest(const EDocumentType documentType, const TVector<TString>& documentNumbers) const {
        return CreateCommonRequest(Config.GetSubscriptionUri(), ERequestMethod::POST, CreateModifySubscribersRequestData(documentType, documentNumbers), ERequestContentType::Json);
    }

    NNeh::THttpRequest TAutocodeFinesClient::CreateRemoveSubscribersRequest(const EDocumentType documentType, const TVector<TString>& documentNumbers) const {
        return CreateCommonRequest(Config.GetUnsubscriptionUri(), ERequestMethod::POST, CreateModifySubscribersRequestData(documentType, documentNumbers), ERequestContentType::Json);
    }

    NNeh::THttpRequest TAutocodeFinesClient::CreateFineConfirmationRequest(const TVector<TString>& externalFineIds) const {
        NJson::TJsonValue payload(NJson::JSON_MAP);
        TJsonProcessor::WriteContainerArray(payload, "idList", externalFineIds);
        return CreateCommonRequest(Config.GetFinesReceivedConfirmationUri(), ERequestMethod::POST, std::move(payload), ERequestContentType::Json);
    }

    NNeh::THttpRequest TAutocodeFinesClient::CreateFineCollectingRequest(const EPenaltyCheckPolicy policy, ui64 limit, ui64 offset) const {
        TString uri;
        if (policy == EPenaltyCheckPolicy::NotPaid) {
            uri = Config.GetFinesUri();
        } else if (policy == EPenaltyCheckPolicy::Paid) {
            uri = Config.GetPaidFinesUri();
        } else {
            Y_UNREACHABLE();
        }

        NJson::TJsonValue payload(NJson::JSON_MAP);
        payload["limit"] = limit;
        payload["offset"] = offset;

        return CreateCommonRequest(uri, ERequestMethod::POST, std::move(payload), ERequestContentType::Json);
    }

    NNeh::THttpRequest TAutocodeFinesClient::CreateViolationPhotoCollectingRequest(const TString& rulingNumber) const {
        NJson::TJsonValue payload(NJson::JSON_MAP);
        payload["rulingNumber"] = rulingNumber;
        return CreateCommonRequest(Config.GetFinePhotosUri(), ERequestMethod::POST, std::move(payload), ERequestContentType::Json);
    }

    // TAutocodeClient

    TAutocodeClient::TAutocodeClient(const TAutocodeClientConfig& config)
        : Config(config)
        , ApiName("autocode-api")
    {
        if (!!Config.GetEvacuationApiConfig()) {
            AutocodeEvacuationClient.Reset(new TAutocodeEvacuationClient(Config, ApiName));
        }

        if (!!Config.GetDriverLicenseApiConfig()) {
            AutocodeDriverLicenseClient.Reset(new TAutocodeDriverLicenseClient(Config, ApiName));
        }

        AutocodeFinesClient.Reset(new TAutocodeFinesClient(Config, ApiName));
    }

    bool TAutocodeClient::CheckIsCarEvacuated(const TString& carNumber, bool& isEvacuated, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!AutocodeEvacuationClient);
        return AutocodeEvacuationClient->CheckIsCarEvacuated(carNumber, isEvacuated, errors);
    }

    bool TAutocodeClient::GetCarEvacuationStatus(const TString& carNumber, TEvacuationInfo& evacuationInfo, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!AutocodeEvacuationClient);
        return AutocodeEvacuationClient->GetCarEvacuationStatus(carNumber, evacuationInfo, errors);
    }

    bool TAutocodeClient::GetDriverLicenseInfo(const TString& licenseNumber, const TInstant issueDate, TDriverLicenseInfo& licenseInfo, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!AutocodeDriverLicenseClient);
        return AutocodeDriverLicenseClient->GetLicenseInfo(licenseNumber, issueDate, licenseInfo, errors);
    }

    bool TAutocodeClient::AddSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const {
        return AutocodeFinesClient->AddSubscribers(documentType, documentNumbers, errors);
    }

    bool TAutocodeClient::RemoveSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const {
        return AutocodeFinesClient->RemoveSubscribers(documentType, documentNumbers, errors);
    }

    bool TAutocodeClient::ConfirmFines(const TVector<TString>& externalFineIds, TMessagesCollector& errors) const {
        return AutocodeFinesClient->ConfirmFines(externalFineIds, errors);
    }

    bool TAutocodeClient::GetFines(TVector<TAutocodeFine>& fines, TMessagesCollector& errors, const EPenaltyCheckPolicy policy, ui64 limit, ui64 offset) const {
        return AutocodeFinesClient->GetFines(fines, errors, policy, limit, offset);
    }

    bool TAutocodeClient::GetViolationPhotos(const TString& rulingNumber, TVector<TAutocodeFinePhoto>& photos, TMessagesCollector& errors) const {
        return AutocodeFinesClient->GetViolationPhotos(rulingNumber, photos, errors);
    }
}
