#pragma once

#include "config.h"
#include "constants.h"
#include "entities.h"
#include "logger.h"

#include <rtline/library/deprecated/async_impl/client.h>

namespace NDrive::NAutocode {
    template <typename TAuthConfig>
    class IAuthProvider {
    public:
        using TPtr = TAtomicSharedPtr<IAuthProvider>;

        IAuthProvider(const TAuthConfig& authConfig)
            : AuthConfig(authConfig)
        {
        }
        virtual ~IAuthProvider() = default;

        virtual void AuthorizeData(NJson::TJsonValue& data) const = 0;

    protected:
        const TAuthConfig& AuthConfig;
    };

    class TAutocodeAuthProvider : public IAuthProvider<TAutocodeClientConfig> {
        using TBase = IAuthProvider<TAutocodeClientConfig>;

    public:
        using TBase::TBase;

        virtual void AuthorizeData(NJson::TJsonValue& data) const override;

    private:
        NJson::TJsonValue GetSecurityData(const TString& seed = "") const;
    };

    template<typename TClientConfig>
    class TAutocodeAuthorizedClientBase : public TRequestClient<TClientConfig, TAutocodeLogger> {
        using TBase = TRequestClient<TClientConfig, TAutocodeLogger>;

    protected:
        TAutocodeAuthorizedClientBase(const TClientConfig& config, const TString& apiName, TAutocodeAuthProvider::TPtr authProvider)
            : TBase(config, apiName)
            , AuthProvider(authProvider)
        {
        }

        NNeh::THttpRequest CreateCommonRequest(const TString& uri, const ERequestMethod method, const NJson::TJsonValue& postData, const ERequestContentType contentType) const {
            NJson::TJsonValue mutablePostData(postData);
            AuthProvider->AuthorizeData(mutablePostData);
            return TBase::CreateCommonRequest(uri, method, mutablePostData, contentType);
        }

        NNeh::THttpRequest CreateCommonRequest(const TString& uri, const ERequestMethod method, NJson::TJsonValue&& postData, const ERequestContentType contentType) const {
            AuthProvider->AuthorizeData(postData);
            return TBase::CreateCommonRequest(uri, method, postData, contentType);
        }

    private:
        TAutocodeAuthProvider::TPtr AuthProvider;
    };

    class TAutocodeEvacuationClient : public TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TEvacuationApiConfig> {
        using TBase = TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TEvacuationApiConfig>;

    public:
        TAutocodeEvacuationClient(const TAutocodeClientConfig& config, const TString& apiName);

        bool CheckIsCarEvacuated(const TString& carNumber, bool& isEvacuated, TMessagesCollector& errors) const;
        bool GetCarEvacuationStatus(const TString& carNumber, TEvacuationInfo& evacuationInfo, TMessagesCollector& errors) const;

    private:
        NNeh::THttpRequest CreateCarEvacuationCheckRequest(const TString& carNumber) const;
    };

    class TAutocodeDriverLicenseClient : public TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TDriverLicenseApiConfig> {
        using TBase = TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TDriverLicenseApiConfig>;

    public:
        TAutocodeDriverLicenseClient(const TAutocodeClientConfig& config, const TString& apiName);

        bool GetLicenseInfo(const TString& licenseNumber, const TInstant issueDate, TDriverLicenseInfo& licenseInfo, TMessagesCollector& errors) const;

    private:
        NNeh::THttpRequest CreateGetLicenseInfoRequest(const TString& licenseNumber, const TInstant issueDate) const;
    };

    class TAutocodeFinesClient : public TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TFinesApiConfig> {
        using TBase = TAutocodeAuthorizedClientBase<TAutocodeClientConfig::TFinesApiConfig>;

    public:
        TAutocodeFinesClient(const TAutocodeClientConfig& config, const TString& apiName);

        bool AddSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const;
        bool RemoveSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const;

        bool ConfirmFines(const TVector<TString>& externalFineIds, TMessagesCollector& errors) const;
        bool GetFines(TVector<TAutocodeFine>& fines, TMessagesCollector& errors, const EPenaltyCheckPolicy policy = EPenaltyCheckPolicy::NotPaid, ui64 limit = 0, ui64 offset = 0) const;

        bool GetViolationPhotos(const TString& rulingNumber, TVector<TAutocodeFinePhoto>& photos, TMessagesCollector& errors) const;

    private:
        bool ProcessModifySubscribersResponse(const NJson::TJsonValue& rawResult, TMessagesCollector& errors) const;
        NJson::TJsonValue CreateModifySubscribersRequestData(const EDocumentType documentType, const TVector<TString>& documentNumbers) const;

        NNeh::THttpRequest CreateAddSubscribersRequest(const EDocumentType documentType, const TVector<TString>& documentNumbers) const;
        NNeh::THttpRequest CreateRemoveSubscribersRequest(const EDocumentType documentType, const TVector<TString>& documentNumbers) const;

        NNeh::THttpRequest CreateFineConfirmationRequest(const TVector<TString>& externalFineIds) const;
        NNeh::THttpRequest CreateFineCollectingRequest(const EPenaltyCheckPolicy policy = EPenaltyCheckPolicy::NotPaid, ui64 limit = 0, ui64 offset = 0) const;
        NNeh::THttpRequest CreateViolationPhotoCollectingRequest(const TString& rulingNumber) const;
    };

    class TAutocodeClient {
    public:
        TAutocodeClient(const TAutocodeClientConfig& config);

        bool HasEvacuationClient() const {
            return !!AutocodeEvacuationClient;
        }

        bool HasDriverLicenseClient() const {
            return !!AutocodeDriverLicenseClient;
        }

        bool CheckIsCarEvacuated(const TString& carNumber, bool& isEvacuated, TMessagesCollector& errors) const;
        bool GetCarEvacuationStatus(const TString& carNumber, TEvacuationInfo& evacuationInfo, TMessagesCollector& errors) const;

        bool GetDriverLicenseInfo(const TString& licenseNumber, const TInstant issueDate, TDriverLicenseInfo& licenseInfo, TMessagesCollector& errors) const;

        bool AddSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const;
        bool RemoveSubscribers(const EDocumentType documentType, const TVector<TString>& documentNumbers, TMessagesCollector& errors) const;

        bool ConfirmFines(const TVector<TString>& externalFineIds, TMessagesCollector& errors) const;
        bool GetFines(TVector<TAutocodeFine>& fines, TMessagesCollector& errors, const EPenaltyCheckPolicy policy = EPenaltyCheckPolicy::NotPaid, ui64 limit = 0, ui64 offset = 0) const;

        bool GetViolationPhotos(const TString& rulingNumber, TVector<TAutocodeFinePhoto>& photos, TMessagesCollector& errors) const;

    private:
        TAutocodeClientConfig Config;
        const TString ApiName;

        THolder<TAutocodeEvacuationClient> AutocodeEvacuationClient;
        THolder<TAutocodeDriverLicenseClient> AutocodeDriverLicenseClient;
        THolder<TAutocodeFinesClient> AutocodeFinesClient;
    };
}
