#include "config.h"

#include <util/stream/file.h>

namespace NDrive::NAutocode {
    TAutocodeClientConfig TAutocodeClientConfig::ParseFromString(const TString& configStr) {
        TAutocodeClientConfig result;
        TAnyYandexConfig config;
        CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
        result.Init(config.GetRootSection(), nullptr);
        return result;
    }

    void TAutocodeClientConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        auto children = section->GetAllChildren();

        {
            auto it = children.find("Evacuation");
            if (it != children.end()) {
                EvacuationApiConfig.Reset(new TEvacuationApiConfig());
                EvacuationApiConfig->Init(it->second, requestPolicy);
            }
        }

        {
            auto it = children.find("DriverLicense");
            if (it != children.end()) {
                DriverLicenseApiConfig.Reset(new TDriverLicenseApiConfig());
                DriverLicenseApiConfig->Init(it->second, requestPolicy);
            }
        }

        {
            auto it = children.find("Fines");
            if (it != children.end()) {
                FinesApiConfig.Reset(new TFinesApiConfig());
                FinesApiConfig->Init(it->second, requestPolicy);
            } else {
                AbortFromCorruptedConfig("Required fines config section does not present in Autocode Api config");
            }
        }

        Secret = section->GetDirectives().Value("Secret", Secret);
        if (!Secret) {
            SecretPath = section->GetDirectives().Value("SecretPath", SecretPath);
            AssertCorrectConfig(SecretPath.Exists(), "Can't read secret from " + SecretPath.GetPath());
            Secret = Strip(TIFStream(SecretPath).ReadAll());
        }
        AssertCorrectConfig(!Secret.empty(), "Autocode Api secret is not set");

        ServiceId = section->GetDirectives().Value("ServiceId", ServiceId);
    }

    void TAutocodeClientConfig::ToString(IOutputStream& os) const {
        if (EvacuationApiConfig) {
            os << "<Evacuation>" << Endl;
            EvacuationApiConfig->ToString(os);
            os << "</Evacuation>" << Endl;
        }
        if (DriverLicenseApiConfig) {
            os << "<DriverLicense>" << Endl;
            DriverLicenseApiConfig->ToString(os);
            os << "</DriverLicense>" << Endl;
        }
        if (FinesApiConfig) {
            os << "<Fines>" << Endl;
            FinesApiConfig->ToString(os);
            os << "</Fines>" << Endl;
        }
        os << "SecretPath: " << SecretPath << Endl;
        os << "ServiceId: " << ServiceId << Endl;
    }

    void TAutocodeClientConfig::TEvacuationApiConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        TBase::Init(section, requestPolicy);
        Uri = section->GetDirectives().Value("Uri", Uri);
    }

    void TAutocodeClientConfig::TEvacuationApiConfig::ToString(IOutputStream& os) const {
        TBase::ToString(os);
        os << "Uri: " << Uri << Endl;
    }

    void TAutocodeClientConfig::TDriverLicenseApiConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        TBase::Init(section, requestPolicy);
        InfoUri = section->GetDirectives().Value("InfoUri", InfoUri);
    }

    void TAutocodeClientConfig::TDriverLicenseApiConfig::ToString(IOutputStream& os) const {
        TBase::ToString(os);
        os << "InfoUri: " << InfoUri << Endl;
    }

    void TAutocodeClientConfig::TFinesApiConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        TBase::Init(section, requestPolicy);
        SubscriptionUri = section->GetDirectives().Value("SubscriptionUri", SubscriptionUri);
        UnsubscriptionUri = section->GetDirectives().Value("UnsubscriptionUri", UnsubscriptionUri);
        FinesUri = section->GetDirectives().Value("FinesUri", FinesUri);
        PaidFinesUri = section->GetDirectives().Value("PaidFinesUri", PaidFinesUri);
        FinePhotosUri = section->GetDirectives().Value("FinePhotosUri", FinePhotosUri);
        FinesReceivedConfirmationUri = section->GetDirectives().Value("FinesReceivedConfirmationUri", FinesReceivedConfirmationUri);
    }

    void TAutocodeClientConfig::TFinesApiConfig::ToString(IOutputStream& os) const {
        TBase::ToString(os);
        os << "SubscriptionUri: " << SubscriptionUri << Endl;
        os << "UnsubscriptionUri " << UnsubscriptionUri << Endl;
        os << "FinesUri: " << FinesUri << Endl;
        os << "PaidFinesUri: " << PaidFinesUri << Endl;
        os << "FinePhotosUri: " << FinePhotosUri << Endl;
        os << "FinesReceivedConfirmationUri: " << FinesReceivedConfirmationUri << Endl;
    }
}
