#include "entities.h"

#include <drive/library/cpp/raw_text/datetime.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <util/datetime/base.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/strip.h>

namespace {
    static const TString DefaultTimezoneName = "Europe/Moscow";
    static const TString DefaultDateFormat = "%d.%m.%Y";
    static const TString DefaultDateTimeFormat = "%d.%m.%Y %H:%M:%S";

    inline bool TryParseDateTime(const NJson::TJsonValue& data, const TString& fieldName, TInstant& instant, bool mustBe, const TString& format, const TString& timezoneName) {
        TString rawDate;
        if (!TJsonProcessor::Read(data, fieldName, rawDate, mustBe)) {
            return false;
        }
        if (!rawDate) {
            return !mustBe;
        }
        instant = NUtil::ParseFomattedLocalDatetime(rawDate, format, timezoneName, TInstant::Zero());
        return true;
    }

    inline bool TryParseDateDefault(const NJson::TJsonValue& data, const TString& fieldName, TInstant& instant, bool mustBe) {
        return TryParseDateTime(data, fieldName, instant, mustBe, DefaultDateFormat, NUtil::utcTzName);
    }

    inline bool TryParseDateTimeDefault(const NJson::TJsonValue& data, const TString& fieldName, TInstant& instant, bool mustBe) {
        return TryParseDateTime(data, fieldName, instant, mustBe, DefaultDateTimeFormat, DefaultTimezoneName);
    }

    template<typename T>
    inline bool TryParseStringOptValue(const NJson::TJsonValue& data, const TString& fieldName, T& result) {
        TString rawValue;
        JREAD_STRING_OPT(data, fieldName, rawValue);
        if (!!rawValue && !TryFromString<T>(rawValue, result)) {
            return false;
        }
        return true;
    }

    template<typename T>
    inline void WriteString(NJson::TJsonValue& data, const TString& fieldName, const T& value) {
        data[fieldName] = (!!value) ? ToString(value) : "";
    }

    inline void WriteDateTime(NJson::TJsonValue& data, const TString& fieldName, const TInstant value, const TString& format) {
        WriteString(data, fieldName, (!!value) ? value.FormatGmTime(format.data()) : "");
    }

    inline void WriteDateDefault(NJson::TJsonValue& data, const TString& fieldName, const TInstant value) {
        WriteDateTime(data, fieldName, value, DefaultDateFormat);
    }
}

namespace NDrive::NAutocode {
    bool TAutocodeFine::DeserializeFromJson(const NJson::TJsonValue& data) {
        JREAD_STRING_NULLABLE_OPT(data, "appealUrl", AppealUrl);
        JREAD_STRING_NULLABLE_OPT(data, "fineInfoUrl", FineInfoUrl);
        JREAD_STRING_NULLABLE_OPT(data, "payUrl", PayUrl);

        TString id;
        JREAD_STRING(data, "id", id);
        if (!TryFromString(id, Id)) {
            return false;
        }

        JREAD_STRING(data, "rulingNumber", RulingNumber);

        if (!TryParseDateDefault(data, "rulingDate", RulingDate, true)) {
            return false;
        }

        JREAD_STRING(data, "articleKoap", ArticleKoap);

        if (!TryParseDateDefault(data, "violationDate", ViolationDate, true)) {
            return false;
        }

        if (!TryParseDateTimeDefault(data, "violationDateWithTime", ViolationDateWithTime, true)) {
            return false;
        }

        JREAD_STRING_NULLABLE_OPT(data, "violationPlace", ViolationPlace);
        JREAD_DOUBLE(data, "sumToPay", SumToPay);

        JREAD_STRING_NULLABLE_OPT(data, "odpsName", OdpsName);
        JREAD_STRING_NULLABLE_OPT(data, "odpsCode", OdpsCode);

        JREAD_STRING_NULLABLE_OPT(data, "violationDocumentType", ViolationDocumentType);

        TString violationDocumentNumber;
        JREAD_STRING(data, "violationDocumentNumber", violationDocumentNumber);
        if (!TryFromString(violationDocumentNumber, ViolationDocumentNumber)) {
            return false;
        }

        JREAD_BOOL(data, "hasPhoto", HasPhoto);

        if (!TryParseDateDefault(data, "discountDate", DiscountDate, false)) {
            return false;
        }

        return true;
    }

    bool TAutocodeFinePhoto::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!data.IsString()) {
            return false;
        }
        try {
            Data = Base64Decode(data.GetString());
        }
        catch (const std::exception& e) {
            return false;
        }
        return true;
    }

    bool TEvacuationInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!data.IsMap() || data.GetMapSafe().empty()) {
            EmptyResult = true;
            Evacuated = false;
            return true;
        }

        EmptyResult = false;

        // all checked cars has the next flag set to true but some don't have fixationDate value
        Evacuated = !data.Has("isVehicleIssued") || !data["isVehicleIssued"].GetBoolean();

        if (!data.Has("violationInfo") || !data["violationInfo"].IsMap()) {
            return false;
        }

        auto violationInfo = data["violationInfo"];
        ViolationArticleCode = violationInfo["articleCode"].GetString();
        ViolationArticleName = violationInfo["articleName"].GetString();
        ViolationArticleTitle = violationInfo["title"].GetString();
        ViolationAddress = violationInfo["address"].GetString();
        if (!TryParseDateTimeDefault(violationInfo, "fixationDate", FixationDate, false)) {
            return false;
        }

        if (data.Has("parkingLotInfo") && data["parkingLotInfo"].IsMap()) {
            ParkingLotAddress = data["parkingLotInfo"]["address"].GetString();
        }

        if (data.Has("departmentInfo") && data["departmentInfo"].IsMap()) {
            DepartmentName = data["departmentInfo"]["name"].GetString();
        }

        return true;
    }

    TString TEvacuationInfo::GetHRReport(const TString& defaultReport) const {
        if (!IsEvacuated()) {
            return defaultReport;
        }
        return (TStringBuilder()
                << "Нарушение: " << ViolationArticleTitle << " " << ViolationArticleCode << " " << ViolationArticleName << "\n"
                << "Адрес: " << ViolationAddress << ", время: " << ((!!FixationDate) ? NUtil::FormatDatetime(FixationDate, DefaultDateFormat) : "неизвестно") << "\n"
                << "Эвакуация: " << DepartmentName << ", стоянка: " << ParkingLotAddress);
    }

    bool TDriverLicenseInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!data.IsMap() || data.GetMap().empty()) {
            EmptyResult = true;
            return true;
        }

        EmptyResult = false;

        if (!TryParseDateDefault(data, "issueDate", IssueDate, false)) {
            return false;
        }
        if (!TryParseDateDefault(data, "endDate", EndDate, false)) {
            return false;
        }

        JREAD_STRING_OPT(data, "issuedOrgan", IssuingState);
        if (!TryParseStringOptValue(data, "drivingExperience", YearExperienceFrom)) {
            return false;
        }
        JREAD_STRING_OPT(data, "allowedCategories", AllowedCategories);

        if (!TryParseDateDefault(data, "deprivationDate", DeprivationDate, false)) {
            return false;
        }
        if (!TryParseStringOptValue(data, "deprivationsCount", DeprivationCount)) {
            return false;
        }

        JREAD_STRING_OPT(data, "status", Status);

        return true;
    }

    NJson::TJsonValue TDriverLicenseInfo::SerializeToJson() const {
        NJson::TJsonValue data(NJson::JSON_MAP);

        // result is intended to be empty if there are no fields initialized
        if (!IsEmptyResult()) {
            WriteDateDefault(data, "issueDate", IssueDate);
            WriteDateDefault(data, "endDate", EndDate);

            WriteString(data, "issuedOrgan", IssuingState);
            WriteString(data, "drivingExperience", YearExperienceFrom);
            WriteString(data, "allowedCategories", AllowedCategories);

            WriteDateDefault(data, "deprivationDate", DeprivationDate);
            WriteString(data, "deprivationsCount", ToString(DeprivationCount));

            WriteString(data, "status", Status);
        }

        return data;
    }
}
