#include <library/cpp/json/json_reader.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/vector.h>
#include <util/string/builder.h>
#include <util/system/env.h>

#include <drive/library/cpp/autocode/entities.h>
#include <drive/library/cpp/autocode/client.h>

Y_UNIT_TEST_SUITE(AutocodeEntities) {
    Y_UNIT_TEST(DeserializeFine) {
        const TString rawFineData = (""
        "["
        "  {"
        "    \"externalUserId\": null,"
        "    \"violationDate\": \"14.12.2019\","
        "    \"rulingDate\": \"14.12.2019\","
        "    \"violationPlace\": \"ДМИТРОВСКОЕ Ш., д.5/1, ТИМИРЯЗЕВСКИЙ (САО) Р-Н, МОСКВА Г.\","
        "    \"payUrl\": \"https://avtokod.mos.ru/fines/payment?uin=d7krp4OrH3386KYD7l4GvBmCvJaqnI7vXQxfD9ohInE%3D&dateDap=14.12.2019&dapNum=18810177191214121324\","
        "    \"articleKoap\": \"12.09.2 - Превышение скорости движения ТС от 20 до 40 км/ч\","
        "    \"discountDate\": \"09.01.2020\","
        "    \"fineInfoUrl\": \"mos.ru/shtrafy/result?dap=18810177191214121324\","
        "    \"odpsCode\": \"45000\","
        "    \"odpsName\": \"УГИБДД ГУ МВД России по г.Москве\","
        "    \"id\": \"41490893\","
        "    \"hasPhoto\": true,"
        "    \"appealUrl\": null,"
        "    \"violationDateWithTime\": \"14.12.2019 00:40:00\","
        "    \"violationDocumentNumber\": \"9916227704\","
        "    \"sumToPay\": 500,"
        "    \"rulingNumber\": \"18810177191214121324\","
        "    \"violationDocumentType\": \"Sts\""
        "  }"
        "]"
        "");

        NJson::TJsonValue finesData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawFineData, &finesData));
        UNIT_ASSERT(finesData.IsArray());

        TVector<NDrive::NAutocode::TAutocodeFine> fines;
        for (const auto& fineData : finesData.GetArray()) {
            NDrive::NAutocode::TAutocodeFine fine;
            UNIT_ASSERT(fine.DeserializeFromJson(fineData));
            fines.push_back(std::move(fine));
        }

        UNIT_ASSERT_VALUES_EQUAL(1, fines.size());

        const auto& fine = fines.front();
        UNIT_ASSERT_VALUES_EQUAL(fine.GetId(), 41490893);
        UNIT_ASSERT_VALUES_EQUAL(fine.GetSumToPay(), 500);
        UNIT_ASSERT_VALUES_EQUAL(fine.GetRulingNumber(), "18810177191214121324");
        UNIT_ASSERT_VALUES_EQUAL(fine.GetViolationDocumentNumber(), 9916227704);
        UNIT_ASSERT_VALUES_EQUAL(fine.GetViolationDateWithTime(), TInstant::ParseIso8601("2019-12-14T00:40:00+0300"));
        UNIT_ASSERT_VALUES_EQUAL(fine.GetDiscountDate(), TInstant::ParseIso8601("2020-01-09T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(fine.GetOdpsName(), "УГИБДД ГУ МВД России по г.Москве");
        UNIT_ASSERT_VALUES_EQUAL(fine.GetArticleKoap(), "12.09.2 - Превышение скорости движения ТС от 20 до 40 км/ч");
        UNIT_ASSERT_VALUES_EQUAL(fine.GetViolationPlace(), "ДМИТРОВСКОЕ Ш., д.5/1, ТИМИРЯЗЕВСКИЙ (САО) Р-Н, МОСКВА Г.");
        UNIT_ASSERT_VALUES_EQUAL(fine.GetRulingDate(), TInstant::ParseIso8601("2019-12-14T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(fine.GetViolationDate(), TInstant::ParseIso8601("2019-12-14T00:00:00Z"));
    }

    Y_UNIT_TEST(DeserializeEvacuationInfo) {
        const TString rawEvacuationInfo = (""
        "{"
        "  \"departmentInfo\": {"
        "    \"name\": \"МАДИ\","
        "    \"phone\": \" 8 (495) 540-76-56\","
        "    \"englishName\": \"MADI\""
        "  },"
        "  \"permissionStatus\": \"Разрешено\","
        "  \"parkingLotInfo\": {"
        "    \"name\": \"Эвакуаторная спецстоянка по адресу ул. Вагоноремонтная, вл. 4А\","
        "    \"workingMode\": \"круглосуточно\","
        "    \"adminArea\": \"г. Москва\","
        "    \"address\": \"г. Москва, ул. Вагоноремонтная, вл. 4А\","
        "    \"district\": \" ул. Вагоноремонтная, вл. 4А\","
        "    \"phones\": ["
        "      \"+7(495)539-5454\""
        "    ],"
        "    \"coordinates\": {"
        "      \"longitude\": \"37.5241669\","
        "      \"latitude\": \"55.9007403\""
        "    }"
        "  },"
        "  \"isVehicleIssued\": true,"
        "  \"violationInfo\": {"
        "    \"articleName\": \"Остановка запрещена\","
        "    \"title\": \"ПДД\","
        "    \"articleCode\": \"3.27\","
        "    \"address\": \"ул. Брестская 2-я\","
        "    \"coordinates\": {"
        "      \"longitude\": \"37.58564611\","
        "      \"latitude\": \"55.77417611\""
        "    },"
        "    \"fixationDate\": \"11.10.2018 15:31:05\","  // optional
        "    \"registrationPlate\": \"К430КА799\""
        "  }"
        "}"
        "");

        NJson::TJsonValue evacuationData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawEvacuationInfo, &evacuationData));

        NDrive::NAutocode::TEvacuationInfo evacuationInfo;
        UNIT_ASSERT(evacuationInfo.DeserializeFromJson(evacuationData));

        UNIT_ASSERT(!evacuationInfo.IsEvacuated());
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetViolationAddress(), "ул. Брестская 2-я");
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetViolationArticleTitle(), "ПДД");
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetViolationArticleCode(), "3.27");
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetViolationArticleName(), "Остановка запрещена");
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetDepartmentName(), "МАДИ");
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetFixationDate(), TInstant::ParseIso8601("2018-10-11T15:31:05+0300"));
        UNIT_ASSERT_VALUES_EQUAL(evacuationInfo.GetParkingLotAddress(), "г. Москва, ул. Вагоноремонтная, вл. 4А");
    }

    Y_UNIT_TEST(DeserializeEmptyEvacuationInfo) {
        NDrive::NAutocode::TEvacuationInfo evacuationInfo;
        UNIT_ASSERT(evacuationInfo.DeserializeFromJson(NJson::JSON_MAP));
        UNIT_ASSERT(!evacuationInfo.IsEvacuated());
    }

    Y_UNIT_TEST(DeserializeLicenseInfoDeprivatedPartial) {
        const TString rawLicenseData = (""
        "{"
        "  \"endDate\": \"29.03.2021\","
        "  \"drivingExperience\": \"\","
        "  \"issuedOrgan\": \"РЭП ГИБДД РУЗСКОГО ОВД\","
        "  \"deprivationsCount\": \"1\","
        "  \"status\": \"Не имеет права управлять ТС\","
        "  \"issueDate\": \"29.03.2011\","
        "  \"allowedCategories\": \"В\","
        "  \"deprivationDate\": \"\""
        "}"
        "");

        NJson::TJsonValue licenseData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawLicenseData, &licenseData));

        NDrive::NAutocode::TDriverLicenseInfo licenseInfo;
        UNIT_ASSERT(licenseInfo.DeserializeFromJson(licenseData));

        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssueDate(), TInstant::ParseIso8601("2011-03-29T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetEndDate(), TInstant::ParseIso8601("2021-03-29T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssuingState(), "РЭП ГИБДД РУЗСКОГО ОВД");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetAllowedCategories(), "В");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetDeprivationCount(), 1);
        UNIT_ASSERT(!licenseInfo.GetYearExperienceFrom());
        UNIT_ASSERT(!licenseInfo.GetDeprivationDate());
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetStatus(), "Не имеет права управлять ТС");
    }

    Y_UNIT_TEST(DeserializeLicenseInfoDeprivatedAllFields) {
        const TString rawLicenseData = (""
        "{"
        "  \"status\": \"Не имеет права управлять ТС\","
        "  \"drivingExperience\": \"2007\","
        "  \"deprivationDate\": \"24.11.2020\","
        "  \"endDate\": \"30.09.2019\","
        "  \"allowedCategories\": \"В\","
        "  \"deprivationsCount\": \"2\","
        "  \"issueDate\": \"31.08.2018\","
        "  \"issuedOrgan\": \"РЭО ОГИБДД МУ МВД РОССИИ БАЛАШИХИНСКОЕ\""
        "}"
        "");

        NJson::TJsonValue licenseData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawLicenseData, &licenseData));

        NDrive::NAutocode::TDriverLicenseInfo licenseInfo;
        UNIT_ASSERT(licenseInfo.DeserializeFromJson(licenseData));

        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssueDate(), TInstant::ParseIso8601("2018-08-31T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetEndDate(), TInstant::ParseIso8601("2019-09-30T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssuingState(), "РЭО ОГИБДД МУ МВД РОССИИ БАЛАШИХИНСКОЕ");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetAllowedCategories(), "В");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetDeprivationCount(), 2);
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetYearExperienceFrom(), 2007);
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetDeprivationDate(), TInstant::ParseIso8601("2020-11-24T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetStatus(), "Не имеет права управлять ТС");
    }

    Y_UNIT_TEST(DeserializeLicenseInfoValidPartial) {
        const TString rawLicenseData = (""
        "{"
        "  \"endDate\": \"10.09.2019\","
        "  \"drivingExperience\": \"\","
        "  \"issuedOrgan\": \"РЭП ГИБДД ХИМКИНСКОГО ОВД\","
        "  \"deprivationsCount\": \"0\","
        "  \"status\": \"Имеет право управлять ТС\","
        "  \"issueDate\": \"10.09.2009\","
        "  \"allowedCategories\": \"В\","
        "  \"deprivationDate\": \"\""
        "}"
        "");

        NJson::TJsonValue licenseData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawLicenseData, &licenseData));

        NDrive::NAutocode::TDriverLicenseInfo licenseInfo;
        UNIT_ASSERT(licenseInfo.DeserializeFromJson(licenseData));

        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssueDate(), TInstant::ParseIso8601("2009-09-10T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetEndDate(), TInstant::ParseIso8601("2019-09-10T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssuingState(), "РЭП ГИБДД ХИМКИНСКОГО ОВД");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetAllowedCategories(), "В");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetDeprivationCount(), 0);
        UNIT_ASSERT(!licenseInfo.GetYearExperienceFrom());
        UNIT_ASSERT(!licenseInfo.GetDeprivationDate());
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetStatus(), "Имеет право управлять ТС");
    }

    Y_UNIT_TEST(DeserializeLicenseInfoValidAllFields) {
        const TString rawLicenseData = (""
        "{"
        "  \"issueDate\": \"12.04.2019\","
        "  \"allowedCategories\": \"ВСM\","
        "  \"deprivationsCount\": \"0\","
        "  \"endDate\": \"12.04.2029\","
        "  \"status\": \"Имеет право управлять ТС\","
        "  \"issuedOrgan\": \"1 ОЭР МО ГИБДД ТНРЭР № 4 ГУ МВД РОССИИ ПО Г. МОСКВ\","
        "  \"drivingExperience\": \"1990\","
        "  \"deprivationDate\": \"\""
        "}"
        "");

        NJson::TJsonValue licenseData;
        UNIT_ASSERT(NJson::ReadJsonFastTree(rawLicenseData, &licenseData));

        NDrive::NAutocode::TDriverLicenseInfo licenseInfo;
        UNIT_ASSERT(licenseInfo.DeserializeFromJson(licenseData));

        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssueDate(), TInstant::ParseIso8601("2019-04-12T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetEndDate(), TInstant::ParseIso8601("2029-04-12T00:00:00Z"));
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetIssuingState(), "1 ОЭР МО ГИБДД ТНРЭР № 4 ГУ МВД РОССИИ ПО Г. МОСКВ");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetAllowedCategories(), "ВСM");
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetDeprivationCount(), 0);
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetYearExperienceFrom(), 1990);
        UNIT_ASSERT(!licenseInfo.GetDeprivationDate());
        UNIT_ASSERT_VALUES_EQUAL(licenseInfo.GetStatus(), "Имеет право управлять ТС");
    }
}
