#include "client.h"

#include <rtline/library/json/parse.h>
#include <library/cpp/xmlrpc/protocol/types.h>

#define ADD_FIELD_4(Scheme, Name, Description, Type) \
    if (!Scheme.HasField(Name)) {                    \
        Scheme.Add<Type>(Name, Description);         \
    }

#define ADD_FIELD_5(Scheme, Name, Description, Required, Type) \
    ADD_FIELD_4(Scheme, Name, Description, Type)               \
    if (Required) {                                            \
        if (auto field = Scheme.Get(Name)) {                   \
            field->SetRequired(true);                          \
        }                                                      \
    }

#define ADD_FIELD_6(Scheme, Name, Description, Required, Length, Type)              \
    ADD_FIELD_5(Scheme, Name, Description, Required, Type)                          \
    if (Length) {                                                                   \
        if (auto field = std::dynamic_pointer_cast<TFSString>(Scheme.Get(Name))) {  \
            field->SetMaxLength(Length);                                            \
        }                                                                           \
    }

#define ADD_FIELD_7(Scheme, Name, Description, Required, Length, Readonly, Type) \
    ADD_FIELD_6(Scheme, Name, Description, Required, Length, Type)               \
    if (Readonly) {                                                              \
        if (auto field = Scheme.Get(Name)) {                                     \
            field->SetReadOnly(true);                                            \
        }                                                                        \
    }



#define ADD_STRING_FIELD(...) Y_PASS_VA_ARGS(Y_MACRO_IMPL_DISPATCHER_6(__VA_ARGS__, ADD_FIELD_7, ADD_FIELD_6, ADD_FIELD_5, ADD_FIELD_4)(__VA_ARGS__, TFSString))


#define ADD_FIELD_VARIANTS(Scheme, Name, Description, Variants)                 \
    ADD_FIELD_4(Scheme, Name, Description, TFSVariants)                         \
    if (auto field = VerifyDynamicCast<TFSVariants*>(Scheme.Get(Name).Get())) { \
        field->SetVariants(Variants);                                           \
    }

void TBalanceClientConfig::Init(const TYandexConfig::Section* section) {
    Host = section->GetDirectives().Value<TString>("Host", Host);
    Uri = section->GetDirectives().Value<TString>("Uri", Uri);
    RequestTimeout = section->GetDirectives().Value<TDuration>("RequestTimeout", RequestTimeout);
    ManagerUid = section->GetDirectives().Value<ui64>("ManagerUid", ManagerUid);
    MethodPrefix = section->GetDirectives().Value<TString>("MethodPrefix", MethodPrefix);
    SelfTvmId = section->GetDirectives().Value<ui64>("SelfTvmId", SelfTvmId);
    DestinationTvmId = section->GetDirectives().Value<ui64>("DestinationTvmId", DestinationTvmId);
    LCProduct = section->GetDirectives().Value<ui64>("LCProduct", LCProduct);
    const TYandexConfig::TSectionsMap children = section->GetAllChildren();
    auto it = children.find("RequestConfig");
    if (it != children.end()) {
        RequestConfig.InitFromSection(it->second);
    }
}

void TBalanceClientConfig::ToString(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Uri: " << Host << Endl;
    os << "RequestTimeout: " << RequestTimeout << Endl;
    os << "ManagerUid: " << ManagerUid << Endl;
    os << "MethodPrefix: " << MethodPrefix << Endl;
    os << "SelfTvmId: " << SelfTvmId << Endl;
    os << "DestinationTvmId: " << DestinationTvmId << Endl;
    os << "LCProduct: " << LCProduct << Endl;
    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
}

TBalanceClient::TBalanceClient(const TBalanceClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
    : Config(config)
    , TvmClient(tvmClient)
{
    Agent = MakeHolder<NNeh::THttpClient>(Config.GetHost(), Config.GetRequestConfig());
}

#define WRITE_OPT(Source, Key, Value) \
    if (Value) {                      \
        Source[Key] = *Value;         \
    }

#define READ_OPT(Source, Key, Value, Type)             \
    if (Source.contains(Key)) {                        \
        Value = NXmlRPC::Cast<Type>(Source.Find(Key)); \
    }

#define READ_REQUIRED(Source, Key, Value, Type)    \
    if (Source.contains(Key)) {                   \
        Value = NXmlRPC::Cast<Type>(Source.Find(Key));          \
    } else if (Source.contains(ToLowerUTF8(Key))) {                 \
        Value = NXmlRPC::Cast<Type>(Source.Find(ToLowerUTF8(Key))); \
    } else if (Source.contains(ToUpperUTF8(Key))) {                 \
        Value = NXmlRPC::Cast<Type>(Source.Find(ToUpperUTF8(Key))); \
    } else {                                                     \
        return false;                                            \
    }

NXmlRPC::TStruct TBalanceClient::TClient::ToXml() const {
    NXmlRPC::TStruct client;
    if (Id) {
        client["CLIENT_ID"] = Id;
    }
    WRITE_OPT(client, "CLIENT_TYPE_ID", Type);
    WRITE_OPT(client, "NAME", Name);
    WRITE_OPT(client, "EMAIL", Email);
    WRITE_OPT(client, "PHONE", Phone);
    WRITE_OPT(client, "FAX", Fax);
    WRITE_OPT(client, "URL", Url);
    WRITE_OPT(client, "AGENCY_ID", AgencyId);
    return client;
}

bool TBalanceClient::TClient::FromXml(const NXmlRPC::TStruct& resultMap) {
    READ_OPT(resultMap, "CLIENT_ID", Id, ui64);
    if (!Id) {
        return false;
    }
    READ_OPT(resultMap, "CLIENT_TYPE_ID", Type, ui64);
    READ_OPT(resultMap, "NAME", Name, TString);
    READ_OPT(resultMap, "EMAIL", Email, TString);
    READ_OPT(resultMap, "PHONE", Phone, TString);
    READ_OPT(resultMap, "FAX", Fax, TString);
    READ_OPT(resultMap, "URL", Url, TString);
    READ_OPT(resultMap, "IS_AGENCY", IsAgency, bool);
    READ_OPT(resultMap, "AGENCY_ID", AgencyId, ui64);
    return true;
}

void TBalanceClient::TClient::AddToJson(NJson::TJsonValue& json) const {
    NJson::InsertNonNull(json, "company", Name);
    NJson::InsertNonNull(json, "email", Email);
    NJson::InsertNonNull(json, "phone", Phone);
    NJson::InsertNonNull(json, "fax", Fax);
}

bool TBalanceClient::TClient::FromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "company", Name)
        && NJson::ParseField(json, "email", Email)
        && NJson::ParseField(json, "phone", Phone)
        && NJson::ParseField(json, "fax", Fax);
}

NDrive::TScheme TBalanceClient::TClient::BuildDefaultScheme(bool update) {
    NDrive::TScheme scheme;
    ADD_STRING_FIELD(scheme, "company", "Название организации", false, 0, update)
    ADD_STRING_FIELD(scheme, "email", "Email")
    ADD_STRING_FIELD(scheme, "phone", "Телефон")
    return scheme;
}

void TBalanceClient::TClient::AddToScheme(NDrive::TScheme& scheme, const NJson::TJsonValue& json, bool update) {
    NDrive::TScheme defaultScheme = BuildDefaultScheme(update);
    defaultScheme.AddToSchemeFromJson(json, {});
    scheme.MergeScheme(defaultScheme);

    if (auto schemeElement = scheme.Get("company"); schemeElement && update) {
        schemeElement->SetReadOnly(update);
    }
}

TExpected<bool, TBalanceClient::TError> TBalanceClient::CreateUserClientAssociation(const TString& uid, const ui64 clientId) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".CreateUserClientAssociation", uid, clientId, uid);
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 2) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }
    return true;
}

TExpected<bool, TBalanceClient::TError> TBalanceClient::RemoveUserClientAssociation(const TString& uid, const ui64 clientId) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".RemoveUserClientAssociation", uid, clientId, uid);
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 2) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }
    return true;
}

TExpected<TVector<TBalanceClient::TClient>, TBalanceClient::TError> TBalanceClient::FindClient(const TString& uid) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".FindClient", NXmlRPC::TStruct().Set("PassportID", uid));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 3) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }

    TVector<TBalanceClient::TClient> clients;
    for (const auto& clientValue : (*result)[2].Array()) {
        TClient client;
        if (!client.FromXml(clientValue.Struct())) {
            TStringStream clientString;
            clientValue.SerializeXml(clientString);
            return MakeUnexpected<TError>({ "Cannot parse client: " + clientString.Str() });
        }
        clients.emplace_back(std::move(client));
    }
    return clients;
}

TExpected<TBalanceClient::TClient, TBalanceClient::TError> TBalanceClient::GetClient(const ui64 id) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetClientByIdBatch", NXmlRPC::TValue(NXmlRPC::TArray(id)));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }

    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 2) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }

    for (const auto& clientValue : (*result)[1].Array()) {
        TClient client;
        TStringStream clientString;
        clientValue.SerializeXml(clientString);
        if (!client.FromXml(clientValue.Struct())) {
            return MakeUnexpected<TError>({ "Cannot parse client: " + clientString.Str() });
        }
        if (client.GetId() != id) {
            return MakeUnexpected<TError>({ "Incorrect client: " + clientString.Str() });
        }
        return client;
    }
    return MakeUnexpected<TError>({ "Empty client result " + stringResult.Str() });
}

TExpected<ui64, TBalanceClient::TError> TBalanceClient::UpdateClient(const TString& uid, const TBalanceClient::TClient& client) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".CreateClient", uid, client.ToXml());
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);
    INFO_LOG << stringResult.Str() << Endl;
    if (result->Size() != 3) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }

    return NXmlRPC::Cast<ui64>((*result)[2]);
}

NXmlRPC::TStruct TBalanceClient::TPerson::ToXml() const {
    NXmlRPC::TStruct person;
    if (Id) {
        person["ID"] = ::ToString(Id);
    }
    person["CLIENT_ID"] = ::ToString(ClientId);
    person["TYPE"] = ::ToString(Type);

    WRITE_OPT(person, "NAME", Name);
    WRITE_OPT(person, "LONGNAME", Longname);
    WRITE_OPT(person, "PHONE", Phone);
    WRITE_OPT(person, "EMAIL", Email);
    WRITE_OPT(person, "POSTCODE", Postcode);
    if (City || Street || Postsuffix) {
        WRITE_OPT(person, "CITY", City);
        WRITE_OPT(person, "STREET", Street);
        WRITE_OPT(person, "POSTSUFFIX", Postsuffix);
    } else {
        WRITE_OPT(person, "POSTADDRESS", Postaddress);
    }
    WRITE_OPT(person, "INN", INN);
    WRITE_OPT(person, "KPP", KPP);
    WRITE_OPT(person, "BIK", BIK);
    WRITE_OPT(person, "ACCOUNT", Account);
    WRITE_OPT(person, "SIGNER_PERSON_NAME", SignerPersonName);
    WRITE_OPT(person, "SIGNER_PERSON_GENDER", SignerPersonGender);
    WRITE_OPT(person, "SIGNER_POSITION_NAME", SignerPositionName);

    WRITE_OPT(person, "LEGAL_ADDRESS_POSTCODE", LegalAddressPostcode);
    WRITE_OPT(person, "LEGALADDRESS", Legaladdress);

    person["DELIVERY_TYPE"] = ::ToString((ui64)DeliveryType);
    return person;
}

bool TBalanceClient::TPerson::FromXml(const NXmlRPC::TStruct& resultMap) {
    READ_OPT(resultMap, "ID", Id, ui64);
    READ_OPT(resultMap, "CLIENT_ID", ClientId, ui64);

    TString typeStr;
    READ_OPT(resultMap, "TYPE", typeStr, TString);
    if (!Id || !ClientId || !TryFromString(typeStr, Type)) {
        return false;
    }

    READ_OPT(resultMap, "NAME", Name, TString);
    READ_OPT(resultMap, "LONGNAME", Longname, TString);
    READ_OPT(resultMap, "PHONE", Phone, TString);
    READ_OPT(resultMap, "EMAIL", Email, TString);
    READ_OPT(resultMap, "POSTCODE", Postcode, TString);
    READ_OPT(resultMap, "POSTADDRESS", Postaddress, TString);
    READ_OPT(resultMap, "POSTSUFFIX", Postsuffix, TString);
    READ_OPT(resultMap, "CITY", City, TString);
    READ_OPT(resultMap, "STREET", Street, TString);
    if (!Postaddress) {
        Postaddress = City.GetOrElse("") + ", " + Street.GetOrElse("") + ", " + Postsuffix.GetOrElse("");
    }
    READ_OPT(resultMap, "INN", INN, TString);
    READ_OPT(resultMap, "KPP", KPP, TString);
    READ_OPT(resultMap, "BIK", BIK, TString);
    READ_OPT(resultMap, "ACCOUNT", Account, TString);
    READ_OPT(resultMap, "SIGNER_PERSON_NAME", SignerPersonName, TString);
    READ_OPT(resultMap, "SIGNER_PERSON_GENDER", SignerPersonGender, TString);
    READ_OPT(resultMap, "SIGNER_POSITION_NAME", SignerPositionName, TString);

    READ_OPT(resultMap, "LEGAL_ADDRESS_POSTCODE", LegalAddressPostcode, TString);
    READ_OPT(resultMap, "LEGALADDRESS", Legaladdress, TString);
    READ_OPT(resultMap, "LEGAL_ADDRESS_SUFFIX", Legaladdress, TString);

    ui64 deliveryType = (ui64)DeliveryType;
    READ_OPT(resultMap, "DELIVERY_TYPE", deliveryType, ui64);
    DeliveryType = (EDeliveryType)deliveryType;

    switch(Type) {
        case EType::Ur:
            return Name
                && Longname
                && Phone
                && Email
                && Postcode
                && (Postaddress || City || Street || Postsuffix)
                && INN
                && BIK
                && Account
                && Legaladdress;
        default:
            return false;
    }
    return true;
}

void TBalanceClient::TPerson::AddToJson(NJson::TJsonValue& json) const {
    NJson::InsertNonNull(json, "company", Name);
    NJson::InsertNonNull(json, "longname", Longname);
    NJson::InsertNonNull(json, "phone", Phone);
    NJson::InsertNonNull(json, "email", Email);
    NJson::InsertNonNull(json, "postcode", Postcode);
    NJson::InsertNonNull(json, "postaddress", Postaddress);
    NJson::InsertNonNull(json, "postsuffix", Postsuffix);
    NJson::InsertNonNull(json, "city", City);
    NJson::InsertNonNull(json, "street", Street);
    NJson::InsertNonNull(json, "inn", INN);
    NJson::InsertNonNull(json, "kpp", KPP);
    NJson::InsertNonNull(json, "bik", BIK);
    NJson::InsertNonNull(json, "account", Account);
    NJson::InsertNonNull(json, "signer_person_name", SignerPersonName);
    NJson::InsertNonNull(json, "signer_person_gender", SignerPersonGender);
    NJson::InsertNonNull(json, "signer_position_name", SignerPositionName);

    NJson::InsertNonNull(json, "legal_address_postcode", LegalAddressPostcode);
    NJson::InsertNonNull(json, "legaladdress", Legaladdress);

    NJson::InsertNonNull(json, "id", Id);
    NJson::InsertNonNull(json, "client_id", ClientId);
    NJson::InsertNonNull(json, "type", ::ToString(Type));
    NJson::InsertNonNull(json, "delivery_type", (ui64)DeliveryType);
}

bool TBalanceClient::TPerson::FromJson(const NJson::TJsonValue& json) {
    TString typeStr;
    if (!NJson::ParseField(json, "type", typeStr) || typeStr && !TryFromString(typeStr, Type)) {
        return false;
    }
    ui64 deliveryType = (ui64)DeliveryType;
    if (!NJson::ParseField(json, "delivery_type", deliveryType, false)) {
        return false;
    }
    if (!NJson::ParseField(json, "inn", INN, Type == EType::Ur) || !NJson::ParseField(json, "kpp", KPP)) {
        return false;
    }
    if (INN && INN->size() == 12 && KPP && !KPP->empty()) {
        return false;
    }
    if (!NJson::ParseField(json, "postaddress", Postaddress, false)
        || !NJson::ParseField(json, "postsuffix", Postsuffix, false)
        || !NJson::ParseField(json, "city", City, false)
        || !NJson::ParseField(json, "street", Street, false))
    {
        return false;
    }
    if (!Postaddress) {
        if (!City && !Street && !Postsuffix) {
            return false;
        }
        Postaddress = City.GetOrElse("") + ", " + Street.GetOrElse("") + ", " + Postsuffix.GetOrElse("");
    }
    DeliveryType = (EDeliveryType)deliveryType;
    return NJson::ParseField(json, "company", Name, Type == EType::Ur)
        && NJson::ParseField(json, "longname", Longname, Type == EType::Ur)
        && NJson::ParseField(json, "phone", Phone, Type == EType::Ur)
        && NJson::ParseField(json, "email", Email, Type == EType::Ur)
        && NJson::ParseField(json, "postcode", Postcode, Type == EType::Ur)
        && NJson::ParseField(json, "bik", BIK, Type == EType::Ur)
        && NJson::ParseField(json, "account", Account, Type == EType::Ur)
        && NJson::ParseField(json, "signer_person_name", SignerPersonName)
        && NJson::ParseField(json, "signer_person_gender", SignerPersonGender)
        && NJson::ParseField(json, "signer_position_name", SignerPositionName)
        && NJson::ParseField(json, "legal_address_postcode", LegalAddressPostcode, Type == EType::Ur)
        && NJson::ParseField(json, "legaladdress", Legaladdress, Type == EType::Ur)
        && NJson::ParseField(json, "id", Id, false)
        && NJson::ParseField(json, "client_id", ClientId, false);
}

NDrive::TScheme TBalanceClient::TPerson::BuildDefaultScheme(const TBalanceClient::TPerson::EType type, bool update, bool separateAddress) {
    NDrive::TScheme scheme;
    {
        auto gTab = scheme.StartTabGuard("Организация");
        ADD_STRING_FIELD(scheme, "company", "Название организации", type == EType::Ur, 0, update)
        ADD_STRING_FIELD(scheme, "longname", "Полное название организации с формой собственности", type == EType::Ur, 0)
        ADD_STRING_FIELD(scheme, "inn", "ИНН", type == EType::Ur, 12, update)
        ADD_STRING_FIELD(scheme, "kpp", "КПП", false, 9)
        ADD_STRING_FIELD(scheme, "legal_address_postcode", "Юридический адрес: почтовый индекс", type == EType::Ur, 6)
        ADD_STRING_FIELD(scheme, "legaladdress", "Юридический адрес: Полный адрес(без индекса)", type == EType::Ur)
    }
    {
        auto gTab = scheme.StartTabGuard("Банковские реквизиты");
        ADD_STRING_FIELD(scheme, "bik", "БИК банка", type == EType::Ur, 9)
        ADD_STRING_FIELD(scheme, "account", "Расчетный счет в банке", type == EType::Ur, 20)
    }
    {
        auto gTab = scheme.StartTabGuard("Данные руководителя");
        ADD_STRING_FIELD(scheme, "signer_person_name", "Имя руководителя", false, 150)
        ADD_STRING_FIELD(scheme, "signer_position_name", "Должность руководителя", false, 150)
    }
    {
        auto gTab = scheme.StartTabGuard("Почтовый адрес для корреспонденции");
        ADD_STRING_FIELD(scheme, "postcode", "Почтовый индекс", type == EType::Ur, 6)
        if (separateAddress) {
            ADD_STRING_FIELD(scheme, "city", "Город", type == EType::Ur, 256)
            ADD_STRING_FIELD(scheme, "street", "Улица", type == EType::Ur, 120)
            ADD_STRING_FIELD(scheme, "postsuffix", "Дом, строение, офис", type == EType::Ur, 250)
        } else {
            ADD_STRING_FIELD(scheme, "postaddress", "Aдрес для доставки корреспонденции/фактический адрес", type == EType::Ur, 0)
        }
    }
    {
        auto gTab = scheme.StartTabGuard("Для связи");
        ADD_STRING_FIELD(scheme, "phone", "Телефон", type == EType::Ur, 256)
        ADD_STRING_FIELD(scheme, "email", "Email", type == EType::Ur, 256)
    }

    return scheme;
}

void TBalanceClient::TPerson::AddToScheme(NDrive::TScheme& scheme, const NJson::TJsonValue& json, const TBalanceClient::TPerson::EType type, bool update, bool separateAddress) {
    auto&& separateAddressFilterRules{separateAddress ? TSet<TString>{"postaddress"} : TSet<TString>{"city", "street", "postsuffix"}};
    NDrive::TScheme defaultScheme = BuildDefaultScheme(type, update, separateAddress);
    defaultScheme.AddToSchemeFromJson(json, separateAddressFilterRules);
    scheme.MergeScheme(defaultScheme);

    TVector<TString> order;
    for (auto&& field : json.GetArray()) {
        auto name = field["name"].GetString();
        if (!separateAddressFilterRules.contains(name)) {
            order.push_back(name);
        }
    }
    scheme.Reorder(order);

    if (auto schemeElement = scheme.Get("company"); schemeElement && update) {
        schemeElement->SetReadOnly(update);
    }

    if (auto schemeElement = scheme.Get("inn"); schemeElement && update) {
        schemeElement->SetReadOnly(update);
    }
}

TExpected<TBalanceClient::TPerson, TBalanceClient::TError> TBalanceClient::GetPerson(const ui64 id) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetPerson", NXmlRPC::TStruct().Set("ID", id));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    for (const auto& personValue : (*result)[0].Array()) {
        TPerson person;
        TStringStream personString;
        personValue.SerializeXml(personString);
        if (!person.FromXml(personValue.Struct())) {
            return MakeUnexpected<TError>({ "Cannot parse person: " + personString.Str() });
        }
        if (person.GetId() != id) {
            return MakeUnexpected<TError>({ "Incorrect person: " + personString.Str() });
        }
        return person;
    }
    return MakeUnexpected<TError>({ "Empty person result " + stringResult.Str() });
}

TExpected<TVector<TBalanceClient::TPerson>, TBalanceClient::TError> TBalanceClient::GetClientPersons(const ui64 id) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetClientPersons", id);
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 1) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }

    TVector<TPerson> persons;
    for (const auto& personValue : (*result)[0].Array()) {
        TPerson person;
        if (!person.FromXml(personValue.Struct())) {
            TStringStream personString;
            personValue.SerializeXml(personString);
            return MakeUnexpected<TError>({ "Cannot parse person: " + personString.Str() });
        }
        if (person.GetClientId() == id) {
            persons.emplace_back(std::move(person));
        }
    }
    return persons;
}

TExpected<ui64, TBalanceClient::TError> TBalanceClient::UpdatePerson(const TString& uid, const TBalanceClient::TPerson& person) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".CreatePerson", uid, person.ToXml());
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    if (result->Size() != 1) {
        TStringStream stringResult;
        result->SerializeXml(stringResult);
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    return NXmlRPC::Cast<ui64>((*result)[0]);
}

NXmlRPC::TStruct TBalanceClient::TContract::ToXml() const {
    NXmlRPC::TStruct contract;
    if (Id) {
        contract["ID"] = Id;
    }
    if (ExternalId) {
        contract["EXTERNAL_ID"] = ExternalId;
    }
    contract["client_id"] = ClientId;
    contract["person_id"] = PersonId;
    contract["currency"] = ::ToString(Currency);
    contract["firm_id"] = FirmId;
    contract["payment_type"] = (ui64)PaymentType;
    contract["manager_uid"] = ManagerUid;
    if (StartTs) {
        contract["start_dt"] = NXmlRPC::TInstantWithoutTimezone(*StartTs);
    }
    contract["services"] = NXmlRPC::TArray();
    for(const auto& service : Services) {
        contract["services"].Array().PushBack(NXmlRPC::TValue(service));
    }
    contract["personal_account"] = (PersonalAccount ? 1 : 0);
    contract["offer_confirmation_type"] = OfferConfirmationType;
    contract["offer_activation_due_term"] = OfferActivationDueTerm;
    contract["offer_activation_payment_amount"] = OfferActivationPaymentAmount;
    if (IsSigned) {
        contract["live_signature"] = 1;
    }
    return contract;
}

bool TBalanceClient::TContract::FromXml(const NXmlRPC::TStruct& resultMap) {
    READ_REQUIRED(resultMap, "ID", Id, ui64)
    READ_REQUIRED(resultMap, "EXTERNAL_ID", ExternalId, TString)
    READ_OPT(resultMap, "CLIENT_ID", ClientId, ui64)
    READ_REQUIRED(resultMap, "PERSON_ID", PersonId, ui64)

    TString currency;
    READ_REQUIRED(resultMap, "CURRENCY", currency, TString)
    if (!currency || !TryFromString(currency, Currency)) {
        return false;
    }
    READ_REQUIRED(resultMap, "FIRM_ID", FirmId, ui64)

    ui64 paymentType = (ui64)PaymentType;
    READ_REQUIRED(resultMap, "PAYMENT_TYPE", paymentType, ui64)
    PaymentType = (EPaymentType)paymentType;
    READ_OPT(resultMap, "DT", StartTs, TInstant)
    READ_OPT(resultMap, "start_ts", StartTs, TInstant)

    Services.clear();
    auto itServices = resultMap.find("SERVICES");
    if (itServices == resultMap.end()) {
        itServices = resultMap.find("services");
    }
    if (itServices == resultMap.end()) {
        return false;
    }
    for(const auto& service: itServices->second.Array()) {
        Services.emplace(NXmlRPC::Cast<ui64>(service));
    }
    READ_OPT(resultMap, "OFFER_CONFIRMATION_TYPE", OfferConfirmationType, TString)
    READ_OPT(resultMap, "OFFER_ACTIVATION_DUE_TERM", OfferActivationDueTerm, ui64)
    READ_OPT(resultMap, "OFFER_ACTIVATION_PAYMENT_AMOUNT", OfferActivationPaymentAmount, ui64)
    READ_OPT(resultMap, "PERSONAL_ACCOUNT", PersonalAccount, bool)
    READ_OPT(resultMap, "IS_ACTIVE", IsActive, bool)
    READ_OPT(resultMap, "IS_SIGNED", IsSigned, bool)
    return true;
}

TExpected<std::pair<ui64, TString>, TBalanceClient::TError> TBalanceClient::CreateOffer(const TString& uid, TBalanceClient::TContract contract) const {
    if (!contract.GetManagerUid()) {
        contract.SetManagerUid(Config.GetManagerUid());
    }
    const auto result = SendRequest(Config.GetMethodPrefix() + ".CreateOffer", uid, contract.ToXml());
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    if (result->Size() != 1) {
        TStringStream stringResult;
        result->SerializeXml(stringResult);
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    return { NXmlRPC::Cast<ui64>((*result)[0]["ID"]), NXmlRPC::Cast<TString>((*result)[0]["EXTERNAL_ID"]) };
}

TExpected<TVector<TBalanceClient::TContract>, TBalanceClient::TError> TBalanceClient::GetContracts(const ui64 clientId) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetClientContracts", NXmlRPC::TStruct().Set("ClientID", clientId));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 1) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }

    TVector<TContract> contracts;
    for (const auto& contractValue : (*result)[0].Array()) {
        TContract contract;
        if (!contract.FromXml(contractValue.Struct())) {
            TStringStream contractString;
            contractValue.SerializeXml(contractString);
            return MakeUnexpected<TError>({ "Cannot parse contract: " + contractString.Str() });
        }
        contracts.emplace_back(std::move(contract));
    }
    return contracts;
}

TExpected<bool, TBalanceClient::TError> TBalanceClient::RestorePassport(const TString& uid) const {
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetPassportByUid", 0, uid, NXmlRPC::TStruct().Set("RepresentedClientIds", true));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    return true;
}

TExpected<std::pair<ui64, bool>, TBalanceClient::TError> TBalanceClient::GetOrCreateContractForUid(const TString& uid, const TBalanceClient::TClient& client, TBalanceClient::TPerson& person, TBalanceClient::TContract& contract) const {
    TVector<ui64> clientIds;
    {
        auto clients = FindClient(uid);
        if (!clients) {
            return MakeUnexpected(clients.GetError());
        }
        for(const auto& client : *clients) {
            clientIds.emplace_back(client.GetId());
        }
    }

    if (clientIds.empty()) {
        auto clientId = UpdateClient(uid, client);
        if (!clientId) {
            return MakeUnexpected(clientId.GetError());
        }
        auto association = CreateUserClientAssociation(uid, *clientId);
        if (!association) {
            return MakeUnexpected(association.GetError());
        }
        clientIds.push_back(*clientId);
    }

    for (const auto& clientId : clientIds) {
        auto existsContracts = GetContracts(clientId);
        if (!existsContracts) {
            return MakeUnexpected(existsContracts.GetError());
        }

        for (const auto& service : contract.GetServices()) {
            for(const auto& existsContract : *existsContracts) {
                if (existsContract.GetServices().contains(service)) {
                    auto balancePerson = GetPerson(existsContract.GetPersonId());
                    if (!balancePerson) {
                        return MakeUnexpected(balancePerson.GetError());
                    }

                    if (balancePerson->HasINN() && person.HasINN() && balancePerson->GetINNRef() != person.GetINNRef()) {
                        TError err("Creating new person with different INN is prohibited");
                        err.SetCode("PERSON_REGISTRATION_WITH_DIFFERENT_INN");
                        return MakeUnexpected<TError>(std::move(err));
                    }
                    
                    person = *balancePerson;
                    contract = existsContract;
                    return { clientId , false };
                }
            }
        }

        person.SetClientId(clientId);
        auto personId = UpdatePerson(uid, person);
        if (!personId) {
            return MakeUnexpected(personId.GetError());
        }
        person.SetId(*personId);

        contract.SetClientId(clientId);
        contract.SetPersonId(*personId);

        auto offerIds = CreateOffer(uid, contract);
        if (!offerIds) {
            return MakeUnexpected(offerIds.GetError());
        }

        contract.SetId(offerIds->first).SetExternalId(offerIds->second);
        return { clientId, true };
    }
    return MakeUnexpected<TError>({ "There is contract for such services" });
}

bool TBalanceClient::TBalance::FromXml(const NXmlRPC::TStruct& resultMap) {
    double receiptSum = 0;
    READ_REQUIRED(resultMap, "ReceiptSum", receiptSum, double);
    ReceiptSum = std::round(receiptSum * 100);

    double actSum = 0;
    READ_REQUIRED(resultMap, "ActSum", actSum, double);
    ActSum = std::round(actSum * 100);

    TMaybe<double> expiredDebtAmount;
    READ_OPT(resultMap, "ExpiredDebtAmount", expiredDebtAmount, double);
    if (expiredDebtAmount) {
        ExpiredDebtAmount = std::round(*expiredDebtAmount * 100);
    }

    READ_OPT(resultMap, "ExpiredDT", ExpiredDT, TInstant);

    TMaybe<double> firstDebtAmount;
    READ_OPT(resultMap, "FirstDebtAmount", firstDebtAmount, double);
    if (firstDebtAmount) {
        FirstDebtAmount = std::round(*firstDebtAmount * 100);
    }

    READ_OPT(resultMap, "FirstDebtFromDT", FirstDebtFromDT, TInstant);
    READ_OPT(resultMap, "FirstDebtPaymentTermDT", FirstDebtPaymentTermDT, TInstant);

    return true;
}

TExpected<TMap<ui64, TBalanceClient::TBalance>, TBalanceClient::TError> TBalanceClient::GetBalance(const ui64 serviceId, const TVector<ui64>& contractIds) const {
    NXmlRPC::TArray contractsInput;
    for (const auto& id : contractIds) {
        contractsInput.PushBack(id);
    }
    const auto result = SendRequest(Config.GetMethodPrefix() + ".GetPartnerBalance", serviceId, contractsInput);
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 1) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    TMap<ui64, TBalance> balances;
    for (const auto& contractValue : (*result)[0].Array()) {
        TBalance balance;
        if (!balance.FromXml(contractValue.Struct())) {
            return MakeUnexpected<TError>({ "Cannot parse " + stringResult.Str() });
        }
        balances[NXmlRPC::Cast<ui64>(contractValue["ContractID"])] = balance;
    }
    return balances;
}

TExpected<bool, TBalanceClient::TError> TBalanceClient::TopupBalance(const ui64 sum, const ui64 partnerId) const {
    TString externalId;
    {
        const auto result = SendRequest(Config.GetMethodPrefix() + ".ExecuteSQL", "balance", "SELECT external_id FROM t_invoice WHERE person_id=" + ::ToString(partnerId));
        if (!result) {
            return MakeUnexpected(result.GetError());
        }
        TStringStream stringResult;
        result->SerializeXml(stringResult);

        if (result->Size() != 1 || (*result)[0].Array().size() != 1) {
            return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
        }

        externalId = NXmlRPC::Cast<TString>((*result)[0][0]["external_id"]);
    }
    const auto result = SendRequest(Config.GetMethodPrefix() + ".MakeOEBSPayment", NXmlRPC::TStruct().Set("InvoiceID", externalId).Set("PaymentSum", sum / 100.));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 1) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }
    ui64 code = NXmlRPC::Cast<ui64>((*result)[0]);
    if (code != 0) {
        return MakeUnexpected<TError>({ "Fail code: " + ::ToString(code) + " " + stringResult.Str() });
    }
    return true;
}

TExpected<TString, TBalanceClient::TError> TBalanceClient::GetPaymentLink(const TString& uid, const ui64 sum, const ui64 clientId, const ui64 contractId) const {
    ui64 orderId = 0;
    ui64 serviceId = 0;
    {
        const auto result = SendRequest(Config.GetMethodPrefix() + ".GetOrdersInfo", NXmlRPC::TArray(NXmlRPC::TStruct().Set("ContractID", contractId)));
        if (!result) {
            return MakeUnexpected(result.GetError());
        }
        TStringStream stringResult;
        result->SerializeXml(stringResult);


        if (result->Size() == 1) {
            for (const auto& order : (*result)[0].Array()) {
                if (NXmlRPC::Cast<ui64>(order["product_id"]) == Config.GetLCProduct()) {
                    orderId = NXmlRPC::Cast<ui64>(order["ServiceOrderID"]);
                    serviceId = NXmlRPC::Cast<ui64>(order["ServiceID"]);
                    break;
                }
            }
            if (!orderId && !serviceId) {
                return MakeUnexpected<TError>({ "No root group " + stringResult.Str() });
            }
        } else {
            return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
        }
    }

    const auto result = SendRequest(Config.GetMethodPrefix() + ".CreateRequest2", uid, clientId, NXmlRPC::TArray(NXmlRPC::TStruct().Set("ServiceID", serviceId).Set("ServiceOrderID", orderId).Set("Qty", sum / 100.)));
    if (!result) {
        return MakeUnexpected(result.GetError());
    }
    TStringStream stringResult;
    result->SerializeXml(stringResult);

    if (result->Size() != 1) {
        return MakeUnexpected<TError>({ "Incorrect result size " + stringResult.Str() });
    }

    return NXmlRPC::Cast<TString>((*result)[0]["UserPath"]);
}

template<>
void Out<TBalanceClient::TError>(IOutputStream& os, const TBalanceClient::TError& error) {
    os << error.GetFullError();
}
