#pragma once

#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/threading/future.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/xmlrpc/protocol/protocol.h>
#include <library/cpp/xmlrpc/protocol/rpcfault.h>
#include <library/cpp/xmlrpc/protocol/value.h>
#include <library/cpp/xmlrpc/protocol/xml.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/deprecated/async_impl/config.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/expected.h>

class TBalanceClientConfig {
    R_READONLY(TString, Host);
    R_READONLY(TString, Uri);
    R_READONLY(ui64, ManagerUid, 1120000000038919);
    R_READONLY(TString, MethodPrefix, "Balance");
    R_READONLY(TDuration, RequestTimeout, TDuration::Seconds(60));
    R_READONLY(ui64, SelfTvmId, 0);
    R_READONLY(ui64, DestinationTvmId, 0);
    R_FIELD(NSimpleMeta::TConfig, RequestConfig);
    R_READONLY(ui64, LCProduct, 511628);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    static TBalanceClientConfig ParseFromString(const TString& configStr) {
        TBalanceClientConfig result;
        TAnyYandexConfig config;
        CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
        result.Init(config.GetRootSection());
        return result;
    }
};

class TBalanceClient {
public:
    class TError {
        R_FIELD(TString, Code, "INTERNAL_ERROR");
        R_FIELD(TString, Message);
        R_FIELD(TString, FullError);
        R_FIELD(ui64, ContractId, 0);

    public:
        TError() = default;
        TError(const TError&) = default;
        TError(const TString& error)
            : FullError(error)
        {}
    };

    class TClient {
        R_FIELD(ui64, Id, 0);
        R_OPTIONAL(int, Type);
        R_OPTIONAL(TString, Name);
        R_OPTIONAL(TString, Email);
        R_OPTIONAL(TString, Phone);
        R_OPTIONAL(TString, Fax);
        R_OPTIONAL(TString, Url);
        R_OPTIONAL(bool, IsAgency);
        R_OPTIONAL(ui64, AgencyId);

    public:
        NXmlRPC::TStruct ToXml() const;
        bool FromXml(const NXmlRPC::TStruct& resultMap);

        void AddToJson(NJson::TJsonValue& json) const;
        bool FromJson(const NJson::TJsonValue& json);
        static NDrive::TScheme BuildDefaultScheme(bool update = false);
        static void AddToScheme(NDrive::TScheme& scheme, const NJson::TJsonValue& json, bool update = false);
    };

    class TPerson {
    public:
        enum class EType {
           Ur /* "ur" */,
        };

        enum class EDeliveryType : ui64 {
           Default,
           Post,
           Yandex,
           Organization,
           VIP,
        };

        R_FIELD(ui64, Id, 0);
        R_FIELD(ui64, ClientId, 0);
        R_FIELD(EType, Type, EType::Ur);

        R_OPTIONAL(TString, Name);
        R_OPTIONAL(TString, Longname);
        R_OPTIONAL(TString, Phone);
        R_OPTIONAL(TString, Email);
        R_OPTIONAL(TString, Postcode);
        R_OPTIONAL(TString, Postaddress);
        R_OPTIONAL(TString, City);
        R_OPTIONAL(TString, Street);
        R_OPTIONAL(TString, Postsuffix);
        R_OPTIONAL(TString, INN);
        R_OPTIONAL(TString, KPP);
        R_OPTIONAL(TString, BIK);
        R_OPTIONAL(TString, Account);
        R_OPTIONAL(TString, SignerPersonName);
        R_OPTIONAL(TString, SignerPersonGender);
        R_OPTIONAL(TString, SignerPositionName);

        R_OPTIONAL(TString, LegalAddressPostcode);
        R_OPTIONAL(TString, Legaladdress);

        R_FIELD(EDeliveryType, DeliveryType, EDeliveryType::Post);

    public:
        NXmlRPC::TStruct ToXml() const;
        bool FromXml(const NXmlRPC::TStruct& resultMap);

        void AddToJson(NJson::TJsonValue& json) const;
        bool FromJson(const NJson::TJsonValue& json);
        static NDrive::TScheme BuildDefaultScheme(const TBalanceClient::TPerson::EType type = TBalanceClient::TPerson::EType::Ur, bool update = false, bool separateAddress = true);
        static void AddToScheme(NDrive::TScheme& scheme, const NJson::TJsonValue& json, const TBalanceClient::TPerson::EType type = TBalanceClient::TPerson::EType::Ur, bool update = false, bool separateAddress = true);
    };

    class TContract {
    public:
        enum class ECurrency {
            RUR,
            UAH,
            BYR,
            BYN,
            KZT,
            USD,
            EUR
        };
        enum class EPaymentType : ui64 {
            UNDEFINED,
            ANY,
            PREPAYMENT,
            POSTPAYMENT,
        };

        R_FIELD(ui64, Id, 0);
        R_FIELD(ui64, ClientId, 0);
        R_FIELD(TString, ExternalId);
        R_FIELD(ECurrency, Currency, ECurrency::RUR);
        R_FIELD(ui64, FirmId, 30);
        R_FIELD(ui64, ManagerUid, 1120000000038919);
        R_FIELD(EPaymentType, PaymentType, EPaymentType::PREPAYMENT);
        R_FIELD(ui64, PersonId, 0);
        R_OPTIONAL(TInstant, StartTs);
        R_FIELD(bool, PersonalAccount, true);
        R_FIELD(TSet<ui64>, Services, { 702 });
        R_OPTIONAL(bool, IsActive);
        R_FIELD(TString, OfferConfirmationType, "min-payment");
        R_FIELD(ui64, OfferActivationDueTerm, 10);
        R_FIELD(ui64, OfferActivationPaymentAmount, 1000);
        R_FIELD(bool, IsSigned, true);

    public:
        TContract() = default;
        TContract(const ui64 clientId, const ui64 personId)
            : ClientId(clientId)
            , PersonId(personId)
        {
        }
        NXmlRPC::TStruct ToXml() const;
        bool FromXml(const NXmlRPC::TStruct& resultMap);
    };

    struct TBalance {
        ui64 ReceiptSum = 0;
        ui64 ActSum = 0;

        bool FromXml(const NXmlRPC::TStruct& resultMap);

        R_OPTIONAL(ui64, ExpiredDebtAmount);
        R_OPTIONAL(TInstant, ExpiredDT);
        R_OPTIONAL(ui64, FirstDebtAmount);
        R_OPTIONAL(TInstant, FirstDebtFromDT);
        R_OPTIONAL(TInstant, FirstDebtPaymentTermDT);
    };

public:
    TBalanceClient(const TBalanceClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient = nullptr);

    TExpected<std::pair<ui64, bool>, TError> GetOrCreateContractForUid(const TString& uid, const TBalanceClient::TClient& client, TBalanceClient::TPerson& person, TBalanceClient::TContract& contract) const;

    TExpected<TMap<ui64, TBalance>, TError> GetBalance(const ui64 serviceId, const TVector<ui64>& contractIds) const;
    TExpected<bool, TError> TopupBalance(const ui64 sum, const ui64 partnerId) const; // only for test

    TExpected<bool, TError> CreateUserClientAssociation(const TString& uid, const ui64 clientId) const;
    TExpected<bool, TError> RemoveUserClientAssociation(const TString& uid, const ui64 clientId) const; // only for test
    TExpected<TVector<TClient>, TError> FindClient(const TString& uid) const;
    TExpected<TClient, TError> GetClient(const ui64 id) const;
    TExpected<ui64, TError> UpdateClient(const TString& uid, const TBalanceClient::TClient& client) const;

    TExpected<TPerson, TError> GetPerson(const ui64 id) const;
    TExpected<TVector<TPerson>, TError> GetClientPersons(const ui64 id) const;
    TExpected<ui64, TError> UpdatePerson(const TString& uid, const TBalanceClient::TPerson& person) const;

    TExpected<std::pair<ui64, TString>, TError> CreateOffer(const TString& uid, TBalanceClient::TContract contract) const;
    TExpected<TVector<TContract>, TError> GetContracts(const ui64 clientId) const;

    TExpected<TString, TError> GetPaymentLink(const TString& uid, const ui64 sum, const ui64 clientId, const ui64 contractId) const;

    TExpected<bool, TError> RestorePassport(const TString& uid) const;

private:
    static NXmlRPC::TArray ParseMultiResult(const TString& res) {
        if (!res) {
            ythrow NXmlRPC::TXmlRPCError() << "empty response";
        }
        NXml::TDocument doc(res, NXml::TDocument::String);
        NXml::TConstNode root(NXmlRPC::FirstChild(doc.Root()));
        if (root.Name() == "params"sv) {
            NXmlRPC::TArray params;
            for (auto node = NXmlRPC::FirstChild(root); !node.IsNull(); node = NXmlRPC::NextSibling(node)) {
                NXmlRPC::Expect(node, "param");
                params.PushBack(NXmlRPC::TValue(NXmlRPC::FirstChild(node)));
            }
            return params;
        }
        ythrow NXmlRPC::TXmlRPCFault(NXmlRPC::TValue(NXmlRPC::FirstChild(root)));
    }

    template <typename... TArgs>
    TExpected<NXmlRPC::TValue, TError> SendRequest(const TStringBuf func, const TArgs&... args) const {
        const TString data = NXmlRPC::SerializeRequest(func, NXmlRPC::TArray(args...));
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUri())
               .SetRequestType("POST")
               .AddHeader("Content-Type", "text/xml")
               .SetPostData(data);

        if (TvmClient) {
            request.AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
        }

        auto responce = Agent->SendAsync(request, Now() + Config.GetRequestTimeout());

        TError error;
        try {
            if (!responce.Wait(Config.GetRequestTimeout()) || !responce.HasValue()) {
                error.SetCode("CLIENT_ERROR");
                error.SetFullError("Timeout");
            } else if (responce.HasException()) {
                error.SetCode("CLIENT_ERROR");
                error.SetFullError(NThreading::GetExceptionMessage(responce));
            } else {
                const auto& reply = responce.GetValue();
                if (!reply.IsSuccessReply()) {
                    error.SetCode("CLIENT_ERROR");
                    error.SetFullError("request error, reply code " + ::ToString(reply.Code()) + ", error: " + reply.ErrorMessage() + ", content: " + reply.Content());
                }
                return NXmlRPC::TArray(ParseMultiResult(reply.Content()));
            }
        } catch (const NXmlRPC::TXmlRPCFault& fault) {
            error.SetFullError(fault.String());
            try {
                NXml::TDocument doc(fault.String(), NXml::TDocument::String);
                NXml::TConstNode root = doc.Root();
                if (root.Name() == "error"sv) {
                    for (auto node = NXmlRPC::FirstChild(root); !node.IsNull(); node = NXmlRPC::NextSibling(node)) {
                        if (node.Name() == "msg") {
                            error.SetMessage(node.Value<TString>());
                        } else if (node.Name() == "code") {
                            error.SetCode(node.Value<TString>());
                        } else if (node.Name() == "contract-id") {
                            error.SetContractId(node.Value<ui64>());
                        }
                    }
                }
            } catch (const yexception& err) {
                ERROR_LOG << err.what() << Endl;
            }
        } catch (const yexception& err) {
            error.SetFullError(err.what());
        }
        return MakeUnexpected<TError>(std::move(error));
    }

private:
    THolder<NNeh::THttpClient> Agent;
    const TBalanceClientConfig Config;
    TAtomicSharedPtr<NTvmAuth::TTvmClient> TvmClient;
};
