#include "client.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/string/builder.h>
#include <util/system/env.h>
#include <util/string/vector.h>

Y_UNIT_TEST_SUITE(BalanceClientSuite) {
    const TString host = "http://greed-ts.paysys.yandex.ru:8002";
    TBalanceClientConfig commonConfig = TBalanceClientConfig::ParseFromString(
            TStringBuilder() << "Host: " << host << Endl
            << "Uri: " << "xmlrpc" << Endl
            << "MethodPrefix: " << "Balance" << Endl
            << "<RequestConfig>" << Endl
            << "MaxAttempts: 1" << Endl
            << "TimeoutSendingms: 1000" << Endl
            << "</RequestConfig>" << Endl
        );

    TBalanceClientConfig testConfig = TBalanceClientConfig::ParseFromString(
            TStringBuilder() << "Host: " << "http://greed-ts.paysys.yandex.ru:30702" << Endl
            << "Uri: " << "xmlrpc" << Endl
            << "MethodPrefix: " << "TestBalance" << Endl
            << "<RequestConfig>" << Endl
            << "MaxAttempts: 1" << Endl
            << "TimeoutSendingms: 1000" << Endl
            << "</RequestConfig>" << Endl
        );


    const TString clientXml = R"(
        <value>
        <struct>
        <member><name>CLIENT_ID</name><value><int>1234</int></value></member>
        <member><name>EMAIL</name><value><string>director@couriers.ru</string></value></member>
        <member><name>NAME</name><value><string>КурСлуб</string></value></member>
        <member><name>PHONE</name><value><string>+74996660666</string></value></member>
        </struct>
        </value>
    )";

    const TString personXml = R"(
        <value><struct>
        <member><name>POSTADDRESS</name><value><string>г. Москва, ул. Новая</string></value></member>
        <member><name>ACCOUNT</name><value><string>40817810055760515502</string></value></member>
        <member><name>PHONE</name><value><string>+74996789065</string></value></member>
        <member><name>SIGNER_POSITION_NAME</name><value><string>ИО</string></value></member>
        <member><name>POSTCODE</name><value><string>360976</string></value></member>
        <member><name>ID</name><value><string>1</string></value></member>
        <member><name>TYPE</name><value><string>ur</string></value></member>
        <member><name>NAME</name><value><string>OAO Ромашка</string></value></member>
        <member><name>LEGAL_ADDRESS_POSTCODE</name><value><string>454544</string></value></member>
        <member><name>SIGNER_PERSON_NAME</name><value><string>ФИО</string></value></member>
        <member><name>CLIENT_ID</name><value><string>123</string></value></member>
        <member><name>LEGAL_ADDRESS_CODE</name><value><string>62021000052</string></value></member>
        <member><name>LEGALADDRESS</name><value><string>п Красно-Андреевский</string></value></member>
        <member><name>LEGAL_ADDRESS_STREET</name><value><string>улица</string></value></member>
        <member><name>LEGAL_ADDRESS_CITY</name><value><string>п Красно-Андреевский</string></value></member>
        <member><name>LONGNAME</name><value><string>Общество с ограниченной ответственностью &quot;Ромашка&quot;</string></value></member>
        <member><name>SIGNER_PERSON_GENDER</name><value><string>Ж</string></value></member>
        <member><name>LEGAL_FIAS_GUID</name><value><string>f30cbbfe-be4a-46e3-bb6e-6a8da907e6cf</string></value></member>
        <member><name>LEGAL_ADDRESS_HOME</name><value><string>дом</string></value></member>
        <member><name>INN</name><value><string>500100732259</string></value></member>
        <member><name>EMAIL</name><value><string>info@romashka.ru</string></value></member>
        <member><name>BIK</name><value><string>044030653</string></value></member>
        </struct></value>
    )";

    const TString contractXml = R"(
        <value><struct>
        <member><name>PERSONAL_ACCOUNT</name><value><boolean>0</boolean></value></member>
        <member><name>FIRM_ID</name><value><int>30</int></value></member>
        <member><name>ID</name><value><int>67</int></value></member>
        <member><name>PERSON_ID</name><value><int>47</int></value></member>
        <member><name>PAYMENT_TYPE</name><value><int>2</int></value></member>
        <member><name>EXTERNAL_ID</name><value><string>5666</string></value></member>
        <member><name>CURRENCY</name><value><string>RUR</string></value></member>
        <member><name>SERVICES</name><value><array><data><value><int>702</int></value></data></array></value></member>
        <member><name>CLIENT_ID</name><value><int>25</int></value></member>
        <member><name>DT</name><value><dateTime.iso8601>20201010T15:11:24</dateTime.iso8601></value></member>
        <member><name>IS_SIGNED</name><value><boolean>1</boolean></value></member>
        </struct></value>
    )";

    TBalanceClient::TClient GetDefaultClient() {
        TBalanceClient::TClient client;
        UNIT_ASSERT(client.FromXml(NXmlRPC::ParseValue(clientXml).Struct()));
        client.SetId({});
        return client;
    }

    TBalanceClient::TPerson GetDefaultPerson() {
        TBalanceClient::TPerson person;
        UNIT_ASSERT(person.FromXml(NXmlRPC::ParseValue(personXml).Struct()));
        person.SetId({});
        person.SetClientId({});
        return person;
    }

    TBalanceClient::TContract GetDefaultContract() {
        TBalanceClient::TContract contract;
        UNIT_ASSERT(contract.FromXml(NXmlRPC::ParseValue(contractXml).Struct()));
        contract.SetId({});
        contract.SetExternalId({});
        contract.SetClientId({});
        contract.SetPersonId({});
        return contract;
    }

    ui64 CreateClient(const TString& uid, const TBalanceClient& balanceClient) {
        TBalanceClient::TClient client = GetDefaultClient();

        auto clientId = balanceClient.UpdateClient(uid, client);
        UNIT_ASSERT_C(clientId, clientId.GetError().GetFullError());
        return *clientId;
    }

    std::pair<ui64, ui64> CreatePerson(const TString& uid, const TBalanceClient& balanceClient) {
        auto clientId = CreateClient(uid, balanceClient);

        TBalanceClient::TPerson person = GetDefaultPerson();
        person.SetClientId(clientId);

        auto personId = balanceClient.UpdatePerson(uid, person);
        UNIT_ASSERT_C(personId, personId.GetError().GetFullError());
        return { clientId, *personId };
    }

    void RemoveAssociations(const TString& uid, const TBalanceClient& balanceClient) {
        auto clients = balanceClient.FindClient(uid);
        UNIT_ASSERT_C(clients, clients.GetError().GetFullError());
        for (const auto& client : *clients) {
            UNIT_ASSERT(balanceClient.RemoveUserClientAssociation(uid, client.GetId()));
        }
        clients = balanceClient.FindClient(uid);
        UNIT_ASSERT(clients && clients->empty());
    }

    void CreateAssociation(const TString& uid, const TBalanceClient& balanceClient) {
        RemoveAssociations(uid, balanceClient);
        UNIT_ASSERT(balanceClient.CreateUserClientAssociation(uid, CreateClient(uid, balanceClient)));
    }

    Y_UNIT_TEST(Client) {
        TBalanceClient::TClient client;
        UNIT_ASSERT(client.FromXml(NXmlRPC::ParseValue(clientXml).Struct()));

        TStringStream ss;
        NXmlRPC::TValue(client.ToXml()).SerializeXml(ss);
        TBalanceClient::TClient clientAfterSerialization;
        UNIT_ASSERT(clientAfterSerialization.FromXml(NXmlRPC::ParseValue(ss.Str()).Struct()));

        UNIT_ASSERT_VALUES_EQUAL(clientAfterSerialization.GetId(), 1234);
        UNIT_ASSERT_VALUES_EQUAL(clientAfterSerialization.GetNameRef(), "КурСлуб");
        UNIT_ASSERT_VALUES_EQUAL(clientAfterSerialization.GetPhoneRef(), "+74996660666");
        UNIT_ASSERT_VALUES_EQUAL(clientAfterSerialization.GetEmailRef(), "director@couriers.ru");
    }

    Y_UNIT_TEST(CreateClient) {
        TBalanceClient balanceClient(commonConfig);
        auto clientId = CreateClient("738314339", balanceClient);
        UNIT_ASSERT(clientId);
        auto externalClient  = balanceClient.GetClient(clientId);
        UNIT_ASSERT_C(externalClient, externalClient.GetError().GetFullError());
    }

    Y_UNIT_TEST(CreateUserClientAssociation) {
        TBalanceClient balanceClient(commonConfig);

        const TString uid = "738314340";
        CreateAssociation(uid, balanceClient);
        auto clients = balanceClient.FindClient(uid);
        UNIT_ASSERT_C(clients && !clients->empty(), clients.GetError().GetFullError());
    }

    Y_UNIT_TEST(Person) {
        TBalanceClient::TPerson person;
        UNIT_ASSERT(person.FromXml(NXmlRPC::ParseValue(personXml).Struct()));

        TStringStream ss;
        NXmlRPC::TValue(person.ToXml()).SerializeXml(ss);

        TBalanceClient::TPerson personAfterSerialization;
        UNIT_ASSERT(personAfterSerialization.FromXml(NXmlRPC::ParseValue(ss.Str()).Struct()));

        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetId(), 1);
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetClientId(), 123);
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetNameRef(), "OAO Ромашка");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetLongnameRef(), "Общество с ограниченной ответственностью \"Ромашка\"");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetPhoneRef(), "+74996789065");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetEmailRef(), "info@romashka.ru");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetPostcodeRef(), "360976");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetPostaddressRef(), "г. Москва, ул. Новая");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetINNRef(), "500100732259");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetBIKRef(), "044030653");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetAccountRef(), "40817810055760515502");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetSignerPersonNameRef(), "ФИО");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetSignerPersonGenderRef(), "Ж");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetSignerPositionNameRef(), "ИО");

        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetLegalAddressPostcodeRef(), "454544");
        UNIT_ASSERT_VALUES_EQUAL(personAfterSerialization.GetLegaladdressRef(), "п Красно-Андреевский");
    }

    Y_UNIT_TEST(CreatePerson) {
        TBalanceClient balanceClient(commonConfig);

        auto [clientId, personId] = CreatePerson("738314339", balanceClient);
        auto externalPerson  = balanceClient.GetPerson(personId);
        UNIT_ASSERT_C(externalPerson, externalPerson.GetError().GetFullError());

        auto clientPersons = balanceClient.GetClientPersons(clientId);
        UNIT_ASSERT_C(clientPersons, clientPersons.GetError().GetFullError());
        UNIT_ASSERT_VALUES_EQUAL(clientPersons->size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(clientPersons->front().GetId(), personId);
    }

    Y_UNIT_TEST(Contract) {
        TBalanceClient::TContract contract;
        UNIT_ASSERT(contract.FromXml(NXmlRPC::ParseValue(contractXml).Struct()));

        TStringStream ss;
        NXmlRPC::TValue(contract.ToXml()).SerializeXml(ss);

        TBalanceClient::TContract contractAfterSerialization;
        UNIT_ASSERT(contractAfterSerialization.FromXml(NXmlRPC::ParseValue(ss.Str()).Struct()));

        UNIT_ASSERT_VALUES_EQUAL(contract.GetId(), 67);
        UNIT_ASSERT_VALUES_EQUAL(contract.GetClientId(), 25);
        UNIT_ASSERT_VALUES_EQUAL(contract.GetExternalId(), "5666");
        UNIT_ASSERT_VALUES_EQUAL(contract.GetFirmId(), 30);
        UNIT_ASSERT_VALUES_EQUAL(contract.GetPersonalAccount(), false);
        UNIT_ASSERT_EQUAL(contract.GetServices(), TSet<ui64>({ 702 }));
        UNIT_ASSERT_VALUES_EQUAL(contract.GetCurrency(), TBalanceClient::TContract::ECurrency::RUR);
        UNIT_ASSERT_VALUES_EQUAL(contract.GetPaymentType(), TBalanceClient::TContract::EPaymentType::PREPAYMENT);
        UNIT_ASSERT_VALUES_EQUAL(contract.GetStartTsRef(), TInstant::ParseIso8601("20201010T15:11:24"));
        UNIT_ASSERT_VALUES_EQUAL(contract.GetIsSigned(), true);
    }

    Y_UNIT_TEST(CreateOffer) {
        TBalanceClient balanceClient(commonConfig);

        auto [clientId, personId] = CreatePerson("738314339", balanceClient);

        TBalanceClient::TContract contract = GetDefaultContract();
        contract.SetClientId(clientId);
        contract.SetPersonId(personId);

        auto offerIds = balanceClient.CreateOffer("738314339", contract);
        UNIT_ASSERT_C(offerIds, offerIds.GetError().GetFullError());

        auto offers = balanceClient.GetContracts(clientId);
        UNIT_ASSERT_C(offers, offers.GetError().GetFullError());
        UNIT_ASSERT_VALUES_EQUAL(offers->size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetId(), offerIds->first);
    }

    Y_UNIT_TEST(CreateContructForNewUid) {
        TBalanceClient balanceClient(commonConfig);
        const TString uid = "738314339";
        RemoveAssociations(uid, balanceClient);

        TBalanceClient::TClient client = GetDefaultClient();
        TBalanceClient::TPerson person = GetDefaultPerson();
        TBalanceClient::TContract contract = GetDefaultContract();

        auto clientId = balanceClient.GetOrCreateContractForUid(uid, client, person, contract);
        UNIT_ASSERT_C(clientId, clientId.GetError().GetFullError());

        auto persons = balanceClient.GetClientPersons(clientId->first);
        UNIT_ASSERT_C(persons, persons.GetError().GetFullError());
        bool found = false;
        for (const auto& personEx : *persons) {
            found |= personEx.GetId() == person.GetId();
        }
        UNIT_ASSERT(found);

        auto offers = balanceClient.GetContracts(clientId->first);
        UNIT_ASSERT_C(offers, offers.GetError().GetFullError());
        UNIT_ASSERT_VALUES_EQUAL(offers->size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetId(), contract.GetId());
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetPersonId(), contract.GetPersonId());
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetPersonId(), person.GetId());
        INFO_LOG << offers->front().GetId() << " " << offers->front().GetExternalId() << Endl;
    }

    Y_UNIT_TEST(CreateContructForExistsUid) {
        TBalanceClient balanceClient(commonConfig);
        const TString uid = "738314339";
        CreateAssociation(uid, balanceClient);

        TBalanceClient::TClient client = GetDefaultClient();
        TBalanceClient::TPerson person = GetDefaultPerson();
        TBalanceClient::TContract contract = GetDefaultContract();

        auto clientId = balanceClient.GetOrCreateContractForUid(uid, client, person, contract);
        UNIT_ASSERT_C(clientId, clientId.GetError().GetFullError());

        auto persons = balanceClient.GetClientPersons(clientId->first);
        UNIT_ASSERT_C(persons, persons.GetError().GetFullError());
        bool found = false;
        for (const auto& personEx : *persons) {
            found |= personEx.GetId() == person.GetId();
        }
        UNIT_ASSERT(found);

        auto offers = balanceClient.GetContracts(clientId->first);
        UNIT_ASSERT_C(offers, offers.GetError().GetFullError());
        UNIT_ASSERT_VALUES_EQUAL(offers->size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetId(), contract.GetId());
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetPersonId(), contract.GetPersonId());
        UNIT_ASSERT_VALUES_EQUAL(offers->front().GetPersonId(), person.GetId());
        INFO_LOG << offers->front().GetId() << " " << offers->front().GetExternalId() << Endl;
    }

    Y_UNIT_TEST(FailedContructForExistsUid) {
        TBalanceClient balanceClient(commonConfig);
        const TString uid = "738314339";
        RemoveAssociations(uid, balanceClient);

        TBalanceClient::TClient client = GetDefaultClient();
        TBalanceClient::TPerson person = GetDefaultPerson();
        TBalanceClient::TContract contract = GetDefaultContract();

        auto clientId = balanceClient.GetOrCreateContractForUid(uid, client, person, contract);
        UNIT_ASSERT_C(clientId, clientId.GetError().GetFullError());

        auto clientId2 = balanceClient.GetOrCreateContractForUid(uid, client, person, contract);
        UNIT_ASSERT_C(clientId2, clientId2.GetError().GetFullError());
        UNIT_ASSERT_EQUAL(clientId->first, clientId2->first);
    }

    void TopupBalance(const ui64 sum, const ui64 personId) {
        TBalanceClient balanceClient(testConfig);
        UNIT_ASSERT(balanceClient.TopupBalance(sum, personId));
    }

    Y_UNIT_TEST(CheckBalance) {
        TBalanceClient balanceClient(commonConfig);

        auto [clientId, personId] = CreatePerson("738314339", balanceClient);

        TBalanceClient::TContract contract = GetDefaultContract();
        contract.SetClientId(clientId);
        contract.SetPersonId(personId);

        auto offerIds = balanceClient.CreateOffer("738314339", contract);
        UNIT_ASSERT_C(offerIds, offerIds.GetError().GetFullError());

        auto zeroBalance = balanceClient.GetBalance(702, { offerIds->first });
        UNIT_ASSERT_C(zeroBalance, zeroBalance.GetError().GetFullError());
        UNIT_ASSERT_EQUAL(zeroBalance->size(), 1);
        auto offerIt = zeroBalance->find(offerIds->first);
        UNIT_ASSERT_UNEQUAL(offerIt, zeroBalance->end());
        UNIT_ASSERT_EQUAL(offerIt->second.ReceiptSum, 0);

        ui64 sum = 532;
        TopupBalance(sum, personId);

        TInstant start = Now();
        ui64 newBalance = offerIt->second.ReceiptSum;
        while(Now() < start + TDuration::Minutes(2) && newBalance != sum) {
            auto balance = balanceClient.GetBalance(702, { offerIds->first });
            UNIT_ASSERT_C(balance, balance.GetError().GetFullError());
            UNIT_ASSERT_EQUAL(balance->size(), 1);
            auto it = balance->find(offerIds->first);
            UNIT_ASSERT_UNEQUAL(it, balance->end());
            newBalance = it->second.ReceiptSum;
        }
        UNIT_ASSERT_EQUAL(sum, newBalance);
    }

    Y_UNIT_TEST(CreatePaymentLink) {
        TBalanceClient balanceClient(commonConfig);

        auto [clientId, personId] = CreatePerson("738314339", balanceClient);

        TBalanceClient::TContract contract = GetDefaultContract();
        contract.SetClientId(clientId);
        contract.SetPersonId(personId);

        auto offerIds = balanceClient.CreateOffer("738314339", contract);
        UNIT_ASSERT_C(offerIds, offerIds.GetError().GetFullError());

        auto link = balanceClient.GetPaymentLink("738314339", 500, clientId, offerIds->first);
        UNIT_ASSERT_C(link, link.GetError().GetFullError());
        UNIT_ASSERT(link->StartsWith("http"));
    }
}
