#include "client.h"

#include <rtline/util/types/exception.h>

#include <library/cpp/blackbox2/src/responseimpl.h> // WTH??
#include <library/cpp/blackbox2/src/xconfig.h> // WTH??
#include <library/cpp/string_utils/url/url.h>

#include <util/string/vector.h>

namespace {
    const TStringBuf InternalBlackbox = "blackbox.yandex-team.ru";
    const TStringBuf ProductionBlackbox = "blackbox.yandex.net";
    const TStringBuf TestingBlackbox = "blackbox-test.yandex.net";
    const TStringBuf TestingBlackbox2 = "pass-test.yandex.ru";

    const NUtil::THttpReply& CheckReply(const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        if (reply.Code() != HTTP_OK) {
            throw TCodedException(reply.Code())
                << "BadResponseCode" << ' '
                << reply.Code() << ' '
                << reply.ErrorMessage() << ' '
                << reply.IsTimeout();
        }
        return reply;
    }

    NTvmAuth::TTvmId DetermineDestinationClientId(TStringBuf endpoint) {
        auto host = GetHost(endpoint);
        if (host == InternalBlackbox) {
            return 223;
        }
        if (host == ProductionBlackbox) {
            return 222;
        }
        if (host == TestingBlackbox || host == TestingBlackbox2) {
            return 224;
        }
        return 0;
    }

    TString DetermineCookieHost(TStringBuf endpoint) {
        auto host = GetHost(endpoint);
        if (host == InternalBlackbox) {
            return "yandex-team.ru";
        }
        if (host == ProductionBlackbox) {
            return "yandex.ru";
        }
        return {};
    }
}

NDrive::TBlackboxClient::TBlackboxClient(TStringBuf endpoint, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm, TAtomicSharedPtr<TAsyncDelivery> ad)
    : HttpClient(MakeAtomicShared<NNeh::THttpClient>(endpoint, NSimpleMeta::TConfig::ForRequester(), ad))
    , Tvm(tvm)
    , DestinationClientId(DetermineDestinationClientId(endpoint))
    , CookieHost(DetermineCookieHost(endpoint))
    , Path(GetPathAndQuery(endpoint).Before('?'))
{
    Options << NBlackbox2::OPT_REGNAME;
    Options << NBlackbox2::OPT_GET_ALL_ALIASES;
    Options << NBlackbox2::TOption("attributes", "1015");
    if (Tvm) {
        Options << NBlackbox2::OPT_GET_ALL_EMAILS;
        Options << NBlackbox2::TOption("getphones", "bound");
        Options << NBlackbox2::TOption("phone_attributes", "102");
    }
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> NDrive::TBlackboxClient::LoginInfoRequest(TStringBuf login, TStringBuf userIp) const {
    TString l(login);
    NBlackbox2::TLoginSid loginSid(l);
    auto request = NBlackbox2::InfoRequest(loginSid, userIp, Options);
    auto reply = MakeRequest(request);
    return reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> TResponsePtr {
        return NBlackbox2::InfoResponse(CheckReply(r).Content());
    });
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> NDrive::TBlackboxClient::UidInfoRequest(TStringBuf uid, TStringBuf userIp) const {
    auto request = NBlackbox2::InfoRequest(uid, userIp, Options);
    auto reply = MakeRequest(request);
    return reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> TResponsePtr {
        return NBlackbox2::InfoResponse(CheckReply(r).Content());
    });
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> NDrive::TBlackboxClient::OAuthRequest(TStringBuf token, TStringBuf userIp) const {
    auto request = NBlackbox2::OAuthRequest(token, userIp, Options);
    auto reply = MakeRequest(request);
    return reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> TResponsePtr {
        return NBlackbox2::SessionIDResponse(CheckReply(r).Content());
    });
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> NDrive::TBlackboxClient::SessionIdRequest(TStringBuf sessionId, TStringBuf userIp) const {
    auto request = NBlackbox2::SessionIDRequest(sessionId, CookieHost, userIp, Options);
    auto reply = MakeRequest(request);
    return reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) -> TResponsePtr {
        return NBlackbox2::SessionIDResponse(CheckReply(r).Content());
    });
}

NDrive::TBlackboxInfo NDrive::TBlackboxClient::Parse(const NBlackbox2::TResponse& response) const {
    NBlackbox2::TUid userId(&response);
    NBlackbox2::TLoginInfo loginInfo(&response);
    NBlackbox2::TAttributes attrs(&response);
    NBlackbox2::TAliasList aliases(&response);
    NBlackbox2::TEmailList emails(&response);
    NBlackbox2::TDisplayNameInfo displayNameInfo(&response);

    TBlackboxInfo info;
    info.Login = loginInfo.Login();
    info.PassportUid = userId.Uid();
    info.IsPlusUser = attrs.Get("1015") == "1";
    for (auto&& item : aliases.GetAliases()) {
        if (item.type() == NBlackbox2::TAliasList::TItem::EType::Yandexoid) {
            info.IsYandexoid = true;
            break;
        }
    }
    if (UserTicketsEnabled) {
        NBlackbox2::TUserTicket userTicket(&response);
        info.TVMTicket = userTicket.Get();
    }
    info.Name = displayNameInfo.Name();
    info.DefaultAvatar = displayNameInfo.DefaultAvatar();

    if (auto responseImpl = response.GetImpl()) {
        auto defaultEmailItem = emails.GetDefaultItem();
        if (!!defaultEmailItem) {
            info.DefaultEmail = defaultEmailItem->Address();
        }

        for (const auto& email : emails.GetEmailItems()) {
            if (email.Validated()) {
                info.ValidatedMails.push_back(email.Address());
            }
        }

        NBlackbox2::xmlConfig::Parts phones(responseImpl->GetParts("phones/phone"));
        if (phones.Size() == 1) {
            TString defaultPhoneStr;
            phones[0].GetIfExists("attribute", defaultPhoneStr);
            if (defaultPhoneStr.size() > 0) {
                info.DefaultPhone = std::move(defaultPhoneStr);
            }
        }

        TString deviceId;
        if (responseImpl->GetIfExists("OAuth/device_id", deviceId)) {
            info.DeviceId = deviceId;
        }

        TString deviceName;
        if (responseImpl->GetIfExists("OAuth/device_name", deviceName)) {
            info.DeviceName = deviceName;
        }

        TString scope;
        if (responseImpl->GetIfExists("OAuth/scope", scope)) {
            info.Scopes = SplitString(scope.data(), scope.size(), " ");
        }
    }
    return info;
}

void NDrive::TBlackboxClient::EnableUserTickets() {
    Options << NBlackbox2::OPT_GET_USER_TICKET;
    UserTicketsEnabled = true;
}

void NDrive::TBlackboxClient::SetDestinationClientId(NTvmAuth::TTvmId value) {
    DestinationClientId = value;
}

void NDrive::TBlackboxClient::SetCookieHost(const TString& value) {
    CookieHost = value;
}

void NDrive::TBlackboxClient::SetScopes(TConstArrayRef<TString> values) {
    Options << NBlackbox2::TOption("scopes", JoinStrings(values.begin(), values.end(), ","));
}

void NDrive::TBlackboxClient::SetExternalOptions(const NBlackbox2::TOptions& extOptions) {
    Options = extOptions;
}

NThreading::TFuture<NUtil::THttpReply> NDrive::TBlackboxClient::MakeRequest(const TString& request) const {
    NNeh::THttpRequest query;
    query.SetUri(Path);
    query.SetCgiData(request);
    if (Tvm) {
        const TString ticket = Tvm->GetServiceTicketFor(DestinationClientId);
        query.AddHeader("X-Ya-Service-Ticket", ticket);
    }
    return HttpClient->SendAsync(query);
}
