#include "calendar_restriction.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/logger/global/global.h>

#include <rtline/util/json_processing.h>

#include <util/generic/array_size.h>
#include <util/generic/serialized_enum.h>
#include <util/string/split.h>

namespace {
    bool ParseDayMinutes(const NJson::TJsonValue& data, const TString& field, ui32& dayMinutes) {
        if (!data.Has(field) || !data[field].IsDefined() || !data[field].GetStringRobust()) {
            return true;
        }

        TString rawInstant;
        if (!TJsonProcessor::Read(data, field, rawInstant)) {
            return false;
        }

        TVector<TString> rawInstantParts = StringSplitter(rawInstant).Split(':');
        if (rawInstantParts.size() != 2) {
            return false;
        }

        ui32 hours, minutes;
        if (!TryFromString(rawInstantParts[0], hours) || !TryFromString(rawInstantParts[1], minutes)) {
            return false;
        }

        dayMinutes = hours * 60 + minutes;
        return true;
    }

    TString SerializeDayMinutes(const ui32 dayMinutes) {
        if (!dayMinutes) {
            return "";
        }
        ui32 hours, minutes;
        hours = dayMinutes / 60;
        minutes = dayMinutes - hours * 60;
        return Sprintf("%02d:%02d", hours, minutes);
    }

    bool ParseYearDay(const NJson::TJsonValue& rawData, ui32& yearDay) {
        TInstant instant;
        if (!rawData.IsString() || !TInstant::TryParseIso8601(rawData.GetString(), instant)) {
            return false;
        }
        yearDay = NUtil::GetYearDay(instant, /* forceLeapYear = */ true);
        return true;
    }

    TString SerializeYearDay(const ui32 yearDay) {
        TInstant instant = TInstant::ParseIso8601("2000-01-01");  // move to a leap year
        instant += TDuration::Days(yearDay);
        return NUtil::FormatDatetime(instant, "%Y-%m-%d");  // reduced isoformat
    }
}

namespace NCalendarRestriction {
    bool TCalendarRestriction::Check(const TInstant instant, TMessagesCollector& foundRestrictions) const {
        TInstant localizedTs = NUtil::ConvertTimeZone(instant, NUtil::GetUTCTimeZone(), GetTimeZone());

        if (!!DayTimeSinceMinutes || !!DayTimeUntilMinutes) {
            auto minutes = NUtil::GetDayMinutes(localizedTs);

            if (!!DayTimeSinceMinutes && minutes < DayTimeSinceMinutes) {
                foundRestrictions.AddMessage("calendar_restrictions_check", "Waiting for start time: too early");
                return false;
            }
            if (!!DayTimeUntilMinutes && minutes >= DayTimeUntilMinutes) {
                foundRestrictions.AddMessage("calendar_restrictions_check", "Waiting for start time: too late");
                return false;
            }
        }

        if (!!DayOfWeekFilter && !DayOfWeekFilter.contains(NUtil::GetWeekDay(localizedTs))) {
            foundRestrictions.AddMessage("calendar_restrictions_check", "Waiting for the next day: filtered by a week day filter");
            return false;
        }

        if (!!HolidayFilter && HolidayFilter.contains(NUtil::GetYearDay(localizedTs, /* forceLeapYear = */ true))) {
            foundRestrictions.AddMessage("calendar_restrictions_check", "Waiting for the next day: today is a holiday");
            return false;
        }

        return true;
    }

    NUtil::TTimeZone TCalendarRestriction::GetDefaultTimeZone() {
        return NUtil::GetTimeZone("Europe/Moscow");
    }

    NDrive::TScheme TCalendarRestriction::GetScheme() {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("time_zone_name", "Таймзона (https://en.wikipedia.org/wiki/List_of_tz_database_time_zones)").SetDefault(TString(GetDefaultTimeZone().name()));
        scheme.Add<TFSString>("day_time_since", "Не раньше чем (ЧЧ:ММ)");
        scheme.Add<TFSString>("day_time_until", "Не позже чем (ЧЧ:ММ)");
        scheme.Add<TFSVariants>("week_days", "Дни запуска").SetVariants(WeekDays).SetMultiSelect(true);
        scheme.Add<TFSArray>("holidays", "Праздничные дни (напр. 2000-03-08)").SetElement<TFSString>();
        return scheme;
    }

    bool TCalendarRestriction::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (data.Has("time_zone_name")) {
            TString timezoneName;
            if (!TJsonProcessor::Read(data, "time_zone_name", timezoneName)) {
                return false;
            }
            if (!NUtil::GetTimeZone(timezoneName, TimeZone)) {
                return false;
            }
        }

        if (!ParseDayMinutes(data, "day_time_since", DayTimeSinceMinutes)) {
            return false;
        }
        if (!ParseDayMinutes(data, "day_time_until", DayTimeUntilMinutes)) {
            return false;
        }

        for (const auto& rawWeekDay : data["week_days"].GetArray()) {
            if (!rawWeekDay.IsString()){
                return false;
            }

            EWeekDay weekDay;
            ui32 weekDayIdx;

            if (TryFromString(rawWeekDay.GetString(), weekDay)) {
                auto weekDayIdxPtr = WeekDaysMapping.FindPtr(weekDay);
                CHECK_WITH_LOG(weekDayIdxPtr != nullptr);
                DayOfWeekFilter.insert(*weekDayIdxPtr);
            } else if (TryFromString(rawWeekDay.GetString(), weekDayIdx) && weekDayIdx < Y_ARRAY_SIZE(WeekDays)) {
                DayOfWeekFilter.insert(weekDayIdx);
            } else {
                return false;
            }
        }

        for (const auto& rawHoliday : data["holidays"].GetArray()) {
            ui32 holidayYearDay;
            if (!ParseYearDay(rawHoliday, holidayYearDay)) {
                return false;
            }
            HolidayFilter.insert(holidayYearDay);
        }

        return true;
    }

    NJson::TJsonValue TCalendarRestriction::SerializeToJson() const {
        NJson::TJsonValue result;

        TJsonProcessor::Write(result, "time_zone_name", TString(TimeZone.name()));

        TString dayTimeSinceSerialized = SerializeDayMinutes(DayTimeSinceMinutes);
        if (!!dayTimeSinceSerialized) {
            TJsonProcessor::Write(result, "day_time_since", dayTimeSinceSerialized);
        }

        TString dayTimeUntilSerialized = SerializeDayMinutes(DayTimeUntilMinutes);
        if (!!dayTimeUntilSerialized) {
            TJsonProcessor::Write(result, "day_time_until", dayTimeUntilSerialized);
        }

        auto& weekDays = result.InsertValue("week_days", NJson::JSON_ARRAY);
        for (const auto& weekDay: DayOfWeekFilter) {
            weekDays.AppendValue(ToString(WeekDays[weekDay]));
        }

        auto& holidays = result.InsertValue("holidays", NJson::JSON_ARRAY);
        for (const auto& yearDay : HolidayFilter) {
            holidays.AppendValue(SerializeYearDay(yearDay));
        }

        return result;
    }
}
