#include "number.h"

#include <util/charset/wide.h>
#include <util/generic/map.h>
#include <util/system/yassert.h>


namespace {
    class TLicensePlateTraits {
    public:
        TLicensePlateTraits() {
            for (auto&&[latin, cyrillic] : RemapToCyrillic) {
                Y_ASSERT(latin < L'z');
                Y_ASSERT(cyrillic >= L'а');
                RemapToLatin.emplace(cyrillic, latin);
            }
        }

        TString ConvertToCyrillic(const TString& value) {
            TUtf16String wide = UTF8ToWide(value);
            for (auto&& c : wide) {
                auto p = RemapToCyrillic.find(c);
                if (p != RemapToCyrillic.end()) {
                    c = p->second;
                }
            }
            return WideToUTF8(wide);
        }

        TString ConvertToLatin(const TString& value) {
            TUtf16String wide = UTF8ToWide(value);
            for (auto&& c : wide) {
                auto p = RemapToLatin.find(c);
                if (p != RemapToLatin.end()) {
                    c = p->second;
                }
            }
            return WideToUTF8(wide);
        }

    private:
        TMap<wchar16, wchar16> RemapToCyrillic = {
            { L'a', L'а' },
            { L'b', L'в' },
            { L'e', L'е' },
            { L'k', L'к' },
            { L'm', L'м' },
            { L'h', L'н' },
            { L'o', L'о' },
            { L'p', L'р' },
            { L'c', L'с' },
            { L't', L'т' },
            { L'y', L'у' },
            { L'x', L'х' },
            { L'A', L'а' },
            { L'B', L'в' },
            { L'E', L'е' },
            { L'K', L'к' },
            { L'M', L'м' },
            { L'H', L'н' },
            { L'O', L'о' },
            { L'P', L'р' },
            { L'C', L'с' },
            { L'T', L'т' },
            { L'Y', L'у' },
            { L'X', L'х' },
        };
        TMap<wchar16, wchar16> RemapToLatin;
    };
}

TString NDrive::ToCyrillic(const TString& licencePlate) {
    return Singleton<TLicensePlateTraits>()->ConvertToCyrillic(licencePlate);
}

TString NDrive::ToLatin(const TString& licencePlate) {
    return Singleton<TLicensePlateTraits>()->ConvertToLatin(licencePlate);
}
