#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/cache.h>

#include <util/string/join.h>


TClearWebClient::TResponse::TResponse(ui32 code, const TString& content)
    : Code(code)
    , Content(content)
{
}

NJson::TJsonValue TClearWebClient::TResponse::GetJsonContent() const {
    NJson::TJsonValue jsonContent;
    return (NJson::ReadJsonFastTree(Content, &jsonContent)) ? jsonContent : NJson::TJsonValue(Content);
}

NJson::TJsonValue TClearWebClient::TResponse::SerializeToJson() const {
    return NJson::TMapBuilder("error_code", GetCode())("error_content", GetJsonContent());
}

TClearWebClient::TClearWebClient(const TClearWebClientConfig& config)
    : Config(config)
    , AD(new TAsyncDelivery())
    , Agent(new NNeh::THttpClient(AD))
{
    AD->Start(Config.GetRequestConfig().GetThreadsStatusChecker(), Config.GetRequestConfig().GetThreadsSenders());
    Agent->RegisterSource("clearweb-client", Config.GetHost(), Config.GetPort(), Config.GetRequestConfig(), true);
}

TClearWebClient::~TClearWebClient() {
    AD->Stop();
}

const TClearWebClientConfig& TClearWebClient::GetConfig() const {
    return Config;
}

class TVerdicts {
    TMap<TString, bool> Results;

public:
    bool Parse(const TString& data) {
        NJson::TJsonValue jsonData;
        if (!NJson::ReadJsonFastTree(data, &jsonData) || !jsonData.Has("result") || !jsonData["result"].Has("verdicts")) {
            return false;
        }
        for (auto&& value : jsonData["result"]["verdicts"].GetArray()) {
            Results.emplace(value["name"].GetStringRobust(), value["value"].GetBooleanRobust());
        }
        return true;
    }

    bool IsValid() const {
        return Results.Value(ToString(EClearWebVerdict::TextAutoGood), false);
    }
};

ui32 TClearWebClient::CheckText(const TString& text, bool& isValid, TMessagesCollector& errors) const {
    auto check = CheckText(text);
    if (!check.Wait(Config.GetRequestTimeout()) || !check.HasValue()) {
        errors.AddMessage("TClearWebClient::CheckText", "Failed by timeout");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    if (check.HasException()) {
        errors.AddMessage("TClearWebClient::CheckText", "Failed with exception: " + NThreading::GetExceptionMessage(check));
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    auto res = check.GetValue();
    TVerdicts verdicts;
    if (!verdicts.Parse(res.GetContent())) {
        errors.AddMessage("TClearWebClient::CheckText", "Fail to parse: " + res.GetContent());
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    isValid = verdicts.IsValid();
    return res.GetCode();
}

NJson::TJsonValue BuildRequestData(const TClearWebClientConfig& config, const TString& text) {
    NJson::TJsonValue params;
    params.InsertValue("type", config.GetDataType());
    params.InsertValue("service", config.GetServiceName());
    auto& body = params.InsertValue("body", NJson::JSON_MAP);
    body.InsertValue(config.GetTestFieldName(), text);
    return NJson::TMapBuilder
        ("jsonrpc", "2.0")
        ("method", "process")
        ("id", "1")
        ("params", params);
}

NThreading::TFuture<TClearWebClient::TResponse> TClearWebClient::CheckText(const TString& text) const {
    NNeh::THttpRequest request;
    request
        .SetUri(Config.GetVersionUri())
        .SetPostData(BuildRequestData(Config, text).GetStringRobust())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json");
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        TUnistatSignalsCache::SignalAdd("clear_web-reply-codes", ToString(reply.Code()), 1);
        return TResponse(reply.Code(), reply.Content());
    });
}
