#include "client.h"

#include <drive/library/cpp/threading/eventlogger.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/storage/sql/query.h>
#include <rtline/util/network/neh.h>

#include <library/cpp/clickhouse/client/columns/date.h>
#include <library/cpp/clickhouse/client/columns/factory.h>
#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/columns/string.h>
#include <library/cpp/clickhouse/client/query.h>
#include <library/cpp/string_utils/quote/quote.h>

namespace {
    struct TMetaItem {
        TString Name;
        TString Type;
    };
    using TMetaItems = TVector<TMetaItem>;

    template <class T>
    void AppendNumeric(NClickHouse::TColumn& column, const NJson::TJsonValue& value) {
        auto impl = column.As<NClickHouse::TColumnVector<T>>();
        Yensured(impl)->Append(NJson::FromJson<T>(value));
    }

    template <class T>
    void AppendString(NClickHouse::TColumn& column, const NJson::TJsonValue& value) {
        auto impl = column.As<T>();
        Yensured(impl)->Append(NJson::FromJson<TString>(value));
    }

    template <class T>
    void AppendTimestamp(NClickHouse::TColumn& column, const NJson::TJsonValue& value) {
        auto impl = column.As<T>();
        auto timestamp = TInstant();
        ui64 v;
        if (TryFromString(value.GetString(), v)) {
            timestamp = TInstant::Seconds(v);
        } else {
            timestamp = NJson::FromJson<TInstant>(value);
        }
        Yensured(impl)->Append(timestamp);
    }

    void Append(NClickHouse::TColumn& column, const NJson::TJsonValue& value) {
        const auto& type = column.Type();
        Y_ENSURE(type);
        switch (type->GetCode()) {
        case NClickHouse::TType::Date:
            AppendTimestamp<NClickHouse::TColumnDate>(column, value);
            break;
        case NClickHouse::TType::DateTime:
            AppendTimestamp<NClickHouse::TColumnDateTime>(column, value);
            break;
        case NClickHouse::TType::Int8:
            AppendNumeric<i8>(column, value);
            break;
        case NClickHouse::TType::Int16:
            AppendNumeric<i16>(column, value);
            break;
        case NClickHouse::TType::Int32:
            AppendNumeric<i32>(column, value);
            break;
        case NClickHouse::TType::Int64:
            AppendNumeric<i64>(column, value);
            break;
        case NClickHouse::TType::UInt8:
            AppendNumeric<ui8>(column, value);
            break;
        case NClickHouse::TType::UInt16:
            AppendNumeric<ui16>(column, value);
            break;
        case NClickHouse::TType::UInt32:
            AppendNumeric<ui32>(column, value);
            break;
        case NClickHouse::TType::UInt64:
            AppendNumeric<ui64>(column, value);
            break;
        case NClickHouse::TType::Float32:
            AppendNumeric<float>(column, value);
            break;
        case NClickHouse::TType::Float64:
            AppendNumeric<double>(column, value);
            break;
        case NClickHouse::TType::String:
            AppendString<NClickHouse::TColumnString>(column, value);
            break;
        case NClickHouse::TType::FixedString:
            AppendString<NClickHouse::TColumnFixedString>(column, value);
            break;
        default:
            ythrow yexception() << "unsupported column type: " << type->GetName();
        }
    }

    template <class T>
    TString GetStringValue(const NClickHouse::TColumn& column, size_t index) {
        Y_ENSURE(index < column.Size(), "index " << index << " is greater than column size " << column.Size());
        auto impl = dynamic_cast<const T*>(&column);
        Y_ENSURE(impl, "cannot cast column to " << TypeName<T>());
        return ToString(impl->At(index));
    }
}

NClickHouse::TAsyncClient::TAsyncClient(const TAsyncClientOptions& options)
    : Options(options)
{
    Client = MakeHolder<NNeh::THttpClient>(Options.Endpoints);
}

NClickHouse::TAsyncClient::~TAsyncClient() {
}

NThreading::TFuture<NClickHouse::TBlock> NClickHouse::TAsyncClient::Execute(TStringBuf query, TInstant deadline) const {
    auto eventlogger = NThreading::GetEventLogger();
    auto request = NNeh::THttpRequest();
    auto cgi = TStringBuilder()
        << "database=" << Options.Database
        << "&date_time_output_format=unix_timestamp"
        << "&query=";
    AppendCgiEscaped(query, cgi);
    if (Options.MaxMemoryUsage) {
        cgi << "&max_memory_usage=" << Options.MaxMemoryUsage;
    }
    request.SetCgiData(cgi);
    request.AddHeader("X-ClickHouse-Format", "JSONCompact");
    request.AddHeader("X-ClickHouse-User", Options.User);
    request.AddHeader("X-ClickHouse-Key", Options.Password);

    auto cutoff = deadline ? deadline : (Now() + Options.Timeout);
    if (eventlogger) {
        eventlogger->AddEvent(NJson::TMapBuilder
            ("event", "ClickHouseAsyncExecute")
            ("database", Options.Database)
            ("deadline", NJson::ToJson(cutoff))
            ("query", query)
        );
    }

    auto reply = Yensured(Client)->SendAsync(request, cutoff);
    auto block = reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();
        const auto parsed = NJson::ReadJsonFastTree(reply.Content());
        const auto meta = NJson::FromJson<TMetaItems>(parsed["meta"]);

        TVector<std::pair<TString, TColumnRef>> columns;
        for (auto&& item : meta) {
            auto column = CreateColumnByType(item.Type);
            Y_ENSURE(column, "cannot create column of type " << item.Type);
            columns.emplace_back(item.Name, column);
        }
        const auto& data = parsed["data"];
        Y_ENSURE(data.IsArray(), "data " << data.GetStringRobust() << " is not an array");
        for (auto&& item : data.GetArray()) {
            Y_ENSURE(item.IsArray(), "item " << item.GetStringRobust() << " is not an array");
            Y_ENSURE(item.GetArray().size() == columns.size(), "item " << item.GetStringRobust() << " has incorrect size, expected " << columns.size());
            for (size_t i = 0; i < columns.size(); ++i) {
                Append(*columns[i].second, item.GetArray()[i]);
            }
        }

        TBlock block;
        for (auto&& [name, column] : columns) {
            block.AppendColumn(name, column);
        }
        return block;
    });
    return block;
}

NThreading::TFuture<NClickHouse::TBlock> NClickHouse::TAsyncClient::Select(TStringBuf table, const NSQL::TQueryOptions& queryOptions, TInstant deadline) const {
    auto query = queryOptions.PrintQuery(TSimpleQuoter::Instance(), table);
    return Execute(query, deadline);
}

template <>
TString NClickHouse::GetColumnValue<TString>(NClickHouse::TColumnRef column, size_t index) {
    Y_ENSURE(column);
    auto type = column->Type();
    Y_ENSURE(type);
    switch (type->GetCode()) {
    case TType::String:
        return GetStringValue<TColumnString>(*column, index);
    case TType::FixedString:
        return GetStringValue<TColumnFixedString>(*column, index);
    default:
        throw yexception() << "cannot cast column of type " << type->GetName() << " to string";
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NClickHouse::TException& object) {
    NJson::TJsonValue result;
    result["code"] = object.Code;
    result["name"] = object.Name;
    if (object.DisplayText) {
        result["text"] = object.DisplayText;
    }
    if (object.StackTrace) {
        result["stacktrace"] = object.StackTrace;
    }
    if (object.Nested) {
        result["nested"] = NJson::ToJson(*object.Nested);
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TMetaItem& result) {
    return
        NJson::TryFromJson(value["name"], result.Name) &&
        NJson::TryFromJson(value["type"], result.Type);
}
