#include "simple.h"

#include <rtline/util/types/cast.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/stream/str.h>
#include <util/stream/zlib.h>
#include <util/system/tls.h>

static Y_THREAD(TString) Compressed;
static Y_THREAD(TString) Debased;

template <>
ZLib::StreamType enum_cast(NDrive::TCompressionOptions::EType value) {
    switch (value) {
    case NDrive::TCompressionOptions::EType::ZLib:
        return ZLib::ZLib;
    case NDrive::TCompressionOptions::EType::GZip:
        return ZLib::GZip;
    }
}

static TString CompressImpl(TStringBuf data, const NDrive::TCompressionOptions& options) {
    if (data.size() > options.CompressionThreshold) {
        TString& compressed = TlsRef(Compressed);
        compressed.clear();
        TStringOutput so(compressed);
        TZLibCompress zo(&so, enum_cast<ZLib::StreamType>(options.Type));
        zo.Write(data);
        zo.Finish();
        return Base64Encode(compressed);
    } else {
        return Base64Encode(data);
    }
}

static TString Debase(TStringBuf data) {
    TString& debased = TlsRef(Debased);
    debased.clear();
    Base64Decode(data, debased);
    return debased;
}

static TString DecompressZLib(const TString& data) {
    TStringInput si(data);
    TZLibDecompress zi(&si);
    return zi.ReadAll();
}

static bool IsZLib(const TString& data) {
    return data.size() > 2 && data[0] == 120 && data[1] == -100;
}

TExpected<TString> NDrive::Compress(TStringBuf data, const TCompressionOptions& options) {
    return WrapUnexpected<yexception>(CompressImpl, data, options);
}

TExpected<TString> NDrive::Decompress(TStringBuf data) {
    auto debased = WrapUnexpected<yexception>(Debase, data);
    if (!debased) {
        return debased;
    }
    if (IsZLib(*debased)) {
        auto decompressed = WrapUnexpected<yexception>(DecompressZLib, *debased);
        if (decompressed) {
            return decompressed;
        }
    }
    return debased;
}
