#pragma once

#include <rtline/util/types/accessor.h>
#include <util/generic/map.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>

template <typename TEntry>
struct TEntryAtomicSharedPtrHolder {
    TEntryAtomicSharedPtrHolder(const TEntry& entry)
        : Entry(MakeAtomicShared<TEntry>(entry))
    {
    }

    TEntryAtomicSharedPtrHolder(TEntry&& entry)
        : Entry(MakeAtomicShared<TEntry>(std::forward<TEntry>(entry)))
    {
    }

    const TEntry& GetEntry() const {
        return *Entry;
    }

    TAtomicSharedPtr<TEntry> Entry;
};

template <typename TEntry>
struct TEntryConstRefHolder {
    TEntryConstRefHolder(const TEntry& entry)
        : Entry(entry)
    {
    }

    const TEntry& GetEntry() const {
        return Entry;
    }

    const TEntry& Entry;
};

struct TFetchContextOptions {
    TString PlaceholderStart = "<";
    TString PlaceholderEnd = ">";
};

template <typename TEntry, typename TEntryHolder = TEntryAtomicSharedPtrHolder<TEntry>>
class TPlainFetchContext {
public:
    using TEntryType = TEntry;
    using TOptions = TFetchContextOptions;
    using TDynamicContext = TMap<TString, TString>;

    explicit TPlainFetchContext(const TEntryType& entry, const TDynamicContext& dynamicContext = {})
        : EntryHolder(entry)
        , DynamicContext(dynamicContext)
    {
    }

    explicit TPlainFetchContext(TEntryType&& entry, const TDynamicContext& dynamicContext = {})
        : EntryHolder(std::forward<TEntryType>(entry))
        , DynamicContext(dynamicContext)
    {
    }

    virtual ~TPlainFetchContext() = default;

    const TEntryType& GetEntry() const {
        return EntryHolder.GetEntry();
    }

private:
    TEntryHolder EntryHolder;

    R_FIELD(TOptions, Options);
    R_FIELD(TDynamicContext, DynamicContext);
};
