#pragma once

#include "context.h"

#include <rtline/util/types/messages_collector.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/object_factory/object_factory.h>
#include <util/generic/algorithm.h>
#include <util/string/join.h>
#include <util/string/split.h>

template <typename TBaseFetcher, typename TContext>
class IContextFetcher {
private:
    enum class ENextBlockAction {
        Default,
        SkipAsConditional,
        FulfillAsConditional
    };

public:
    using TContextType = TContext;
    using TEntryType = typename TContextType::TEntryType;

    using TPtr = TAtomicSharedPtr<TBaseFetcher>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<TBaseFetcher, TString, TString, TVector<TString>>;

    IContextFetcher() = default;
    explicit IContextFetcher(TString originalPlaceholderName, TVector<TString> parameters);
    virtual ~IContextFetcher() = default;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const = 0;

    static TString ProcessText(const TString& text, const TContextType& context, TMessagesCollector& errors);
    static TSet<TString> GetRegisteredFetchers();

    template <typename TFetcherImpl>
    static TString GetPlaceholderName(const TContextType& context) {
        return EnclosePlaceholder(TFetcherImpl::GetTypeName(), context);
    }

protected:
    TString OriginalPlaceholderName;
    TVector<TString> Parameters;

protected:
    NJson::TJsonValue GetCombinedContextField(const TContextType& context, TStringBuf name) const;

private:
    static TString Process(const TString& placeholderName, const TVector<TString>& placeholderParameters, const TContextType& context, TMessagesCollector& errors);

    static TString StripPlaceholder(TStringBuf text, const TContextType& context, size_t lo, size_t hi);
    static TString EnclosePlaceholder(TStringBuf content, const TContextType& context);

    static size_t FindNextPlaceholderStart(TStringBuf text, const TContextType& context, size_t prevPlaceholderEndPosition);
    static size_t FindNextPlaceholderEnd(TStringBuf text, const TContextType& context, size_t currentPlaceholderStartPosition, bool& hasNested);

    static std::tuple<TString, TVector<TString>> ProcessPlaceholder(TStringBuf placeholder);
};

template <typename TBaseFetcher, typename TContext>
IContextFetcher<TBaseFetcher, TContext>::IContextFetcher(TString originalPlaceholderName, TVector<TString> parameters)
    : OriginalPlaceholderName(originalPlaceholderName)
    , Parameters(std::move(parameters))
{
}

template <typename TBaseFetcher, typename TContext>
TString IContextFetcher<TBaseFetcher, TContext>::ProcessText(const TString& text, const TContextType& context, TMessagesCollector& errors) {
    TString result;
    size_t lo = 0, hi = 0;

    ENextBlockAction nextBlockAction = ENextBlockAction::Default;

    while ((lo = FindNextPlaceholderStart(text, context, hi)) != TString::npos) {
        result += TStringBuf(text, hi, lo - hi);

        bool hasNested = false;
        if ((hi = FindNextPlaceholderEnd(text, context, lo, hasNested)) != TString::npos) {
            hi += context.GetOptions().PlaceholderEnd.length();

            if (nextBlockAction == ENextBlockAction::SkipAsConditional) {
                nextBlockAction = ENextBlockAction::Default;
                continue;
            }

            TString placeholderContent = StripPlaceholder(text, context, lo, hi);

            if (nextBlockAction == ENextBlockAction::FulfillAsConditional) {
                if (hasNested) {
                    placeholderContent = ProcessText(placeholderContent, context, errors);
                }
                result += placeholderContent;
                nextBlockAction = ENextBlockAction::Default;
                continue;
            }

            const bool isConditional = placeholderContent.Contains('=');
            if (isConditional) {
                bool conditionMatched = false;

                TVector<TStringBuf> conditions = StringSplitter(placeholderContent).Split('|');
                for (auto&& condition: conditions) {
                    TVector<TStringBuf> conditionParts = StringSplitter(condition).Split('=').Take(2);

                    if (hasNested) {
                        conditionMatched = (conditionParts.size() == 2 && ProcessText(TString{conditionParts[0]}, context, errors) == ProcessText(TString{conditionParts[1]}, context, errors));
                    } else {
                        conditionMatched = (conditionParts.size() == 2 && conditionParts[0] == conditionParts[1]);
                    }

                    if (conditionMatched) {
                        break;
                    }
                }

                nextBlockAction = (conditionMatched) ? ENextBlockAction::FulfillAsConditional : ENextBlockAction::SkipAsConditional;
                continue;
            }

            TString placeholderName;
            TVector<TString> placeholderParameters;
            std::tie(placeholderName, placeholderParameters) = ProcessPlaceholder(placeholderContent);

            if (hasNested) {
                const auto transformer = [&](auto&& item){ return ProcessText(item, context, errors); };
                placeholderName = transformer(placeholderName);
                Transform(placeholderParameters.begin(), placeholderParameters.end(), placeholderParameters.begin(), transformer);
            }

            result += Process(placeholderName, placeholderParameters, context, errors);
            nextBlockAction = ENextBlockAction::Default;
        } else {
            hi = lo;
            break;
        }
    }

    if (hi != TString::npos) {
        result += TStringBuf(text, hi, text.length() - hi);
    }

    return result;
}

template <typename TBaseFetcher, typename TContext>
TString IContextFetcher<TBaseFetcher, TContext>::StripPlaceholder(TStringBuf text, const TContextType& context, size_t lo, size_t hi) {
    const size_t contentStart = lo + context.GetOptions().PlaceholderStart.length();
    const size_t contentEnd = hi - context.GetOptions().PlaceholderEnd.length();
    return TString(TStringBuf(text, contentStart, contentEnd - contentStart));
}

template <typename TBaseFetcher, typename TContext>
TString IContextFetcher<TBaseFetcher, TContext>::EnclosePlaceholder(TStringBuf content, const TContextType& context) {
    return JoinSeq("", { context.GetOptions().PlaceholderStart, content, context.GetOptions().PlaceholderEnd });
}

template <typename TBaseFetcher, typename TContext>
size_t IContextFetcher<TBaseFetcher, TContext>::FindNextPlaceholderStart(TStringBuf text, const TContextType& context, size_t prevPlaceholderEndPosition) {
    return text.find(context.GetOptions().PlaceholderStart, prevPlaceholderEndPosition);  // do not add unit to process string leading char correctly
}

template <typename TBaseFetcher, typename TContext>
size_t IContextFetcher<TBaseFetcher, TContext>::FindNextPlaceholderEnd(TStringBuf text, const TContextType& context, size_t currentPlaceholderStartPosition, bool& hasNested) {
    size_t placeholderStartCount = 1;

    size_t nextPlaceholderStart = text.find(context.GetOptions().PlaceholderStart, currentPlaceholderStartPosition + 1);
    size_t nextPlaceholderEnd = text.find(context.GetOptions().PlaceholderEnd, currentPlaceholderStartPosition + 1);

    while (nextPlaceholderEnd != TString::npos) {
        if (nextPlaceholderStart < nextPlaceholderEnd) {
            if (++placeholderStartCount > 1) {
                hasNested = true;
            }
            nextPlaceholderStart = text.find(context.GetOptions().PlaceholderStart, nextPlaceholderStart + 1);
        } else if (nextPlaceholderEnd < nextPlaceholderStart) {
            if (--placeholderStartCount == 0) {
                break;
            }
            nextPlaceholderEnd = text.find(context.GetOptions().PlaceholderEnd, nextPlaceholderEnd + 1);
        } else {
            break;  // positions could be equal only TString::npos
        }
    }

    return (placeholderStartCount == 0 && nextPlaceholderEnd != TString::npos) ? nextPlaceholderEnd : TString::npos;
}

template <typename TBaseFetcher, typename TContext>
TSet<TString> IContextFetcher<TBaseFetcher, TContext>::GetRegisteredFetchers() {
    TSet<TString> keys;
    TFactory::GetRegisteredKeys(keys);
    return keys;
}

template <typename TBaseFetcher, typename TContext>
TString IContextFetcher<TBaseFetcher, TContext>::Process(const TString& placeholderName, const TVector<TString>& placeholderParameters, const TContextType& context, TMessagesCollector& errors) {
    if (!placeholderName) {
        return "";  // remove empty placeholders
    }

    if (!TFactory::Has(placeholderName)) {
        const auto* valuePtr = context.GetDynamicContext().FindPtr(placeholderName);
        if (valuePtr != nullptr) {
            return *valuePtr;
        }
    }

    TPtr fetcher;
    if (TFactory::Has(placeholderName)) {
        fetcher = TFactory::Construct(placeholderName, placeholderName, placeholderParameters);
    } else if (TFactory::Has("default")) {
        fetcher = TFactory::Construct("default", placeholderName, placeholderParameters);
    } else {
        fetcher = nullptr;
    }

    if (fetcher) {
        TString result;
        if (fetcher->Fetch(context, result, errors)) {
            return result;
        }
    }

    return EnclosePlaceholder(placeholderName, context);  // envelope placeholder back without parameters
}

template <typename TBaseFetcher, typename TContext>
std::tuple<TString, TVector<TString>> IContextFetcher<TBaseFetcher, TContext>::ProcessPlaceholder(TStringBuf placeholder) {
    size_t parametersStart, parametersEnd;
    if ((parametersStart = placeholder.find('(')) != TString::npos &&
        (parametersEnd = placeholder.find(')', parametersStart)) == (placeholder.length() - 1)
    ) {
        TString placeholderName = TString(placeholder.SubString(0, parametersStart));  // strip parameters start characters
        TVector<TString> parameters = StringSplitter(placeholder.SubString(1 + parametersStart, parametersEnd - (1 + parametersStart))).Split(',');
        return { placeholderName, std::move(parameters) };
    }
    return { TString(placeholder), {} };
}

template <typename TBaseFetcher, typename TContext>
NJson::TJsonValue  IContextFetcher<TBaseFetcher, TContext>::GetCombinedContextField(const TContextType& context, TStringBuf name) const {
    NJson::TJsonValue value;
    if (context.GetEntry()[name].IsDefined()) {
        value = context.GetEntry()[name];
    } else if (auto valuePtr = context.GetDynamicContext().FindPtr(name)) {
        value = *valuePtr;
    }
    return value;
}
