#include <drive/library/cpp/context_fetcher/fetcher.h>

#include <drive/library/cpp/raw_text/datetime.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

#include <util/charset/utf8.h>
#include <util/datetime/base.h>
#include <util/system/env.h>

namespace {
    class TTestFetchContext: public TPlainFetchContext<NJson::TJsonValue> {
        using TBase = TPlainFetchContext<NJson::TJsonValue>;

    public:
        using TBase::TBase;
    };

    class ITestContextFetcher: public IContextFetcher<ITestContextFetcher, TTestFetchContext> {
        using TBase = IContextFetcher<ITestContextFetcher, TTestFetchContext>;

    public:
        using TBase::TBase;
    };

    class TDefaultTestContextFetcher: public ITestContextFetcher {
        using TBase = ITestContextFetcher;

    public:
        using TBase::TBase;

        bool Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const override {
            auto* propertyPtr = context.GetEntry().GetValueByPath(OriginalPlaceholderName);
            if (!propertyPtr || !propertyPtr->IsDefined()) {
                return false;
            }

            if (propertyPtr->IsBoolean()) {
                if (Parameters.size() >= 1 && ::ToLowerUTF8(Parameters[0]) == "eng") {
                    result = (propertyPtr->GetBoolean()) ? "Yes" : "No";
                } else {
                    result = (propertyPtr->GetBoolean()) ? "Да" : "Нет";
                }
            } else {
                // default parser cannot detect type (besides boolean), so type should be specified explicitly
                if (Parameters.size() >= 1) {
                    const TString& parameterType = Parameters[0];
                    if (parameterType.StartsWith("instant")) {
                        TInstant timestamp;
                        if (!TryFromJson(*propertyPtr, timestamp)) {
                            return false;
                        }
                        if (parameterType.Contains("_isoformat")) {
                            result = timestamp.ToString();
                        } else {
                            if (parameterType.Contains("_local")) {
                                timestamp = NUtil::ConvertTimeZone(timestamp, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone("Europe/Moscow"));
                            }
                            result = NUtil::FormatDatetime(timestamp, "%d.%m.%Y %H:%M");
                        }
                    }
                } else {
                    result = propertyPtr->GetStringRobust();
                }
            }

            return true;
        }

        static TString GetTypeName() {
            return "default";
        }

    private:
        using TRegistrator = TFactory::TRegistrator<TDefaultTestContextFetcher>;
        static TRegistrator Registrator;
    };

    TDefaultTestContextFetcher::TRegistrator TDefaultTestContextFetcher::Registrator(TDefaultTestContextFetcher::GetTypeName());
}

Y_UNIT_TEST_SUITE(Context) {
    Y_UNIT_TEST(RegisteredFetchers) {
        auto registeredFetchers = ITestContextFetcher::GetRegisteredFetchers();
        UNIT_ASSERT_EQUAL(registeredFetchers.size(), 1);
        UNIT_ASSERT_STRINGS_EQUAL(*registeredFetchers.begin(), "default");
    }

    Y_UNIT_TEST(AbsentField) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_name_absent>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, templateToProcess);
    }

    Y_UNIT_TEST(PresentField) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_name>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "42");
    }

    Y_UNIT_TEST(PresentTextField) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", "42");
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_name>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "42");
    }

    Y_UNIT_TEST(InstantLocalFieldDefault) {
        // default parser cannot detect type (besides boolean), so type should be specified explicitly
        NJson::TJsonValue entry = NJson::TMapBuilder("some_instant", 1597000000);  // 2020-08-09T19:06:40+00:00
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_instant(instant_local)>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "09.08.2020 22:06");
    }

    Y_UNIT_TEST(InstantStringFieldDefault) {
        // default parser cannot detect type (besides boolean), so type should be specified explicitly
        NJson::TJsonValue entry = NJson::TMapBuilder("some_instant", "2020-08-09T22:06:40+03:00");  // 2020-08-09T19:06:40+00:00
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_instant(instant)>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "09.08.2020 19:06");
    }

    Y_UNIT_TEST(InstantFieldIsoFormat) {
        // default parser cannot detect type (besides boolean), so type should be specified explicitly
        NJson::TJsonValue entry = NJson::TMapBuilder("some_instant", 1597000000);  // 2020-08-09T19:06:40+00:00
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_instant(instant_isoformat)>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "2020-08-09T19:06:40.000000Z");
    }

    Y_UNIT_TEST(PresentMultiple) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42)("some_other_name", 64);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "values are <some_name> and <some_other_name>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "values are 42 and 64");
    }

    Y_UNIT_TEST(NestedFieldName) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42)("some_42", 84);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<some_<some_name>>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "84");
    }

    Y_UNIT_TEST(NestedSimple) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "<prefix <some_name> suffix>";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "<prefix 42 suffix>");
    }

    Y_UNIT_TEST(ConditionalMatched) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "prefix <<some_name>=42><some_text> suffix";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix some_text suffix");
    }

    Y_UNIT_TEST(ConditionalNotMatched) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "prefix <<some_name>=21><some_text> suffix";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix  suffix");
    }

    Y_UNIT_TEST(AnyOfConditionsMatched) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "prefix <<some_name>=42|<some_name>=43><some_text> suffix";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix some_text suffix");
    }

    Y_UNIT_TEST(AllConditionsNotMatched) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", 42);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess = "prefix <<some_name>=21|<some_name>=22><some_text> suffix";

        TString processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix  suffix");
    }

    Y_UNIT_TEST(ParameterizedSimple) {
        NJson::TJsonValue entry = NJson::TMapBuilder("some_name", true);
        TTestFetchContext context(entry);
        TMessagesCollector errors;

        TString templateToProcess, processedValue;

        templateToProcess = "prefix <some_name(Eng)> suffix";
        processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix Yes suffix");

        templateToProcess = "prefix <some_name(Rus)> suffix";
        processedValue = ITestContextFetcher::ProcessText(templateToProcess, context, errors);
        UNIT_ASSERT_STRINGS_EQUAL(processedValue, "prefix Да suffix");
    }
}
