#include "client.h"

#include <rtline/library/json/adapters.h>

#include <util/charset/utf8.h>
#include <util/stream/str.h>

bool TryParseLocalDateTime(const TString& date, TInstant& result, TString& error) {
    TDateTimeFields dateField;
    TVector<TString> dateTime(SplitString(date, " "));
    if (dateTime.size() > 0) {
        TVector<TString> splitted(SplitString(dateTime[0], "."));;
        if (splitted.size() != 3) {
            error = "Unexpected date format (dd.mm.yyyy) " + date;
            return false;
        }
        bool dateRead = TryFromString(splitted[0], dateField.Day);
        dateRead &= TryFromString(splitted[1], dateField.Month);
        dateRead &= TryFromString(splitted[2], dateField.Year);
        if (!dateRead) {
            error = "Couldn't parse date fields " + date;
            return false;
        }
    }
    if (dateTime.size() > 1) {
        TVector<TString> splitted(SplitString(dateTime[1], ":"));;
        if (splitted.size() != 3) {
            error = "Unexpected time format (hh:mm:ss) " + date;
            return false;
        }
        bool timeRead = TryFromString(splitted[0], dateField.Hour);
        timeRead &= TryFromString(splitted[1], dateField.Minute);
        timeRead &=TryFromString(splitted[2], dateField.Second);
        if (!timeRead) {
            error = "Couldn't parse time fields " + date;
            return false;
        }
    }
    if (!dateField.IsOk()) {
        error = "Couldn't convert date (dd.mm.yyyy hh:mm:ss) " + date;
        return false;
    }
    result = dateField.ToInstant(TInstant::Zero());
    return result != TInstant::Zero();
}

template <class T>
bool TryGetAttributeByKey(const NJson::TJsonValue::TArray& array, const TSet<TString>& keyOptions, T& result) {
    TString keyValue;
    for (auto&& el : array) {
        if (ParseField(el["key"], keyValue, true) &&  keyOptions.contains(keyValue)) {
            if (ParseField(el["value"], result, true)) {
                return true;
            }
        }
    }
    return false;
}

TString TokenizeString(size_t from, size_t to, size_t skip, TChar token, const TString& str) {
    TString result;
    TStringStream ss;
    to = Min(to, str.size());
    size_t cur = 1;
    for (size_t i = 0; i < str.size(); i++) {
        ss << str[i];
        if (i >= from && cur++ % skip == 0 && i < to - 1) {
            ss << token;
        }
    }
    return ss.Str();
}

void ParseJsonReply(const NJson::TJsonValue& replyJson, const TInstant minUntil, TCovidPassClient::TPassData& passDataResult) {
    passDataResult.SetApiResponse(replyJson);
    passDataResult.SetIsValid(false);
    int valid;
    if (!NJson::ParseField(replyJson["valid"], valid, true)) {
        passDataResult.SetError("No 'valid' field in result json");
        return;
    } else if (valid == 0) {
        return;
    }
    TInstant created;
    TString error;
    TString untilString;
    TInstant validUntil = TInstant::Zero();
    if (TryGetAttributeByKey(replyJson["parameters"].GetArray(), {"Действует по", "Действует до", "по"}, untilString)) {
        TInstant validUntilParam;
        if (TryParseLocalDateTime(untilString, validUntilParam, error) || TInstant::TryParseIso8601(untilString, validUntilParam)) {
            validUntil = validUntilParam;
        } else {
            validUntil = TInstant::Zero();
            passDataResult.SetError(passDataResult.GetError() + "\nError parsing 'valid until' data: " + error);
        }
    }
    if (validUntil == TInstant::Zero()) {
        TString createdString;
        if (!NJson::ParseField(replyJson["created"], createdString, true)) {
            passDataResult.SetError("No 'created' field in result json");
        } else if (!TInstant::TryParseIso8601(createdString, created)
                    && !TryParseLocalDateTime(createdString, created, error)) {
            passDataResult.SetError("Error parsing created time. " + error);
            created = TInstant::Zero();
        }
        if (created == TInstant::Zero() && TryGetAttributeByKey(replyJson["parameters"].GetArray(), {"Действителен с"}, createdString)) {
            TInstant createdParam;
            if (TryParseLocalDateTime(createdString, createdParam, error) || TInstant::TryParseIso8601(createdString, created)) {
                created = createdParam;
            } else {
                passDataResult.SetError(passDataResult.GetError() + "\nError parsing 'valid from' data: " + error);
            }
        }
        if (created == TInstant::Zero()) {
            return;
        }
        TDuration duration = TDuration::Zero();
        ui64 minutes = 0, hours = 0, days = 0;
        if (!replyJson["hours"].IsDefined() && !replyJson["minutes"].IsDefined()) {
            if (replyJson["parameters"].IsArray()) {
                TString untilString;
                if (TryGetAttributeByKey(replyJson["parameters"].GetArray(), {"Время действия пропуска(ч)"}, hours)) {
                    duration += TDuration::Hours(hours);
                }
                if (TryGetAttributeByKey(replyJson["parameters"].GetArray(), {"Время действия пропуска(д)"}, days)) {
                    duration += TDuration::Days(days);
                }
            }
        } else if (NJson::ParseField(replyJson["hours"], hours) || NJson::ParseField(replyJson["minutes"], minutes)) {
            duration += TDuration::Hours(hours);
            duration += TDuration::Minutes(minutes);
        }
        if (duration == TDuration::Zero()) {
            passDataResult.SetError(passDataResult.GetError() + "\nNo pass duration info");
            return;
        }
        passDataResult.SetError("");
        validUntil = created + duration;
    }

    passDataResult.SetValidUntil(validUntil);
    if (validUntil >= minUntil) {
        passDataResult.SetIsValid(true);
    }
    return;
}

bool TCovidPassMoscowRegionClient::CheckAndUpdateToken(TString& error) const {
    if (TInstant::Now() < TokenValidUntil && !Token.empty()) {
        return true;
    }

    if (!RefreshToken.empty()) {
        NJson::TJsonValue payload = NJson::JSON_MAP;
        payload["refreshToken"] = RefreshToken;
        if (UpdateToken(payload, "api/v2/account/refreshToken", error)) {
            return true;
        }
    }
    NJson::TJsonValue payload = NJson::JSON_MAP;
    payload["login"] = Config.GetLogin();
    payload["password"] = Config.GetPassword();
    return UpdateToken(payload, "api/v2/account/login", error);
}

bool TCovidPassMoscowRegionClient::UpdateToken(const NJson::TJsonValue& payload, const TString& uri, TString& error) const {
    NNeh::THttpRequest request;
    request
        .SetUri(uri)
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json")
        .SetPostData(payload.GetStringRobust());

    NUtil::THttpReply reply = Agent->SendMessageSync(request, Now() + Config.GetRequestTimeout());
    NJson::TJsonValue replyJson;
    if (reply.Code() == 200 && NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
        if (!NJson::ParseField(replyJson["refreshToken"]["value"], RefreshToken)) {
            RefreshToken = "";
        }
        TString expiresIn;
        if (NJson::ParseField(replyJson["accessToken"]["expires"], expiresIn, true)) {
            if (!TInstant::TryParseIso8601(expiresIn, TokenValidUntil)) {
                TokenValidUntil = TInstant::Zero();
            }
        }
        if (NJson::ParseField(replyJson["accessToken"]["value"], Token, true)) {
            return true;
        }
    } else {
        Token = "";
        if (reply.Code() != 200) {
            error = "Bad reply, code " + ToString(reply.Code()) + ". Error: " + reply.ErrorMessage();
        } else {
            error = "Broken json. " + reply.Content();
        }
    }
    return false;
}

bool TCovidPassMoscowClient::CheckAndUpdateToken(TString& error) const {
    if (TInstant::Now() < TokenValidUntil) {
        return true;
    }
    TString grantType = "grant_type=client_credentials";
    TString reqData = Base64Encode(Config.GetKey() + ":" + Config.GetSecret());
    NNeh::THttpRequest request;
    request
        .SetUri("token")
        .SetRequestType("POST")
        .AddHeader("Authorization", "Basic " + reqData)
        .AddHeader("Content-Type", "application/x-www-form-urlencoded")
        .SetPostData(grantType);
    TInstant now = TInstant::Now();
    NUtil::THttpReply reply = Agent->SendMessageSync(request, Now() + Config.GetRequestTimeout());
    NJson::TJsonValue replyJson;
    if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
        error = "Bad reply, code " + ToString(reply.Code()) + ". Error: " + reply.ErrorMessage();
        return false;
    }
    ui32 expiresIn = 0;
    if (!NJson::ParseField(replyJson["expires_in"], expiresIn) || !NJson::ParseField(replyJson["access_token"], Token)) {
        error = "No access token info in reply";
        Token = "";
        return false;
    }
    TokenValidUntil = now + TDuration::Seconds(expiresIn);
    return true;
}

NThreading::TFuture<TCovidPassClient::TPassData> TCovidPassMoscowClient::GetPassData(const TString& pass, const TInstant minUntil) const {
    TString error;
    if (!CheckAndUpdateToken(error)) {
        TCovidPassClient::TPassData passData;
        passData.SetError(error);
        passData.SetIsValid(false);
        return NThreading::MakeFuture(passData);
    }

    NJson::TJsonValue payload = NJson::JSON_MAP;
    payload["id"] = pass;
    NNeh::THttpRequest request;
    request
        .SetUri(Config.GetApiPath())
        .SetRequestType("POST")
        .AddHeader("Authorization", "Bearer " + Token)
        .AddHeader("Content-Type", "application/json")
        .SetPostData(payload.GetStringRobust());

    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([pass, minUntil](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        TCovidPassClient::TPassData passData;
        passData.SetIsValid(false);
        passData.SetPassToken(pass);
        NJson::TJsonValue replyJson;
        if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            passData.SetError("Bad response, code " + ToString(reply.Code()) + ". Error: " + reply.ErrorMessage());
            passData.SetApiResponse(replyJson);
            return passData;
        }
        passData.SetApiResponse(replyJson);
        if (!replyJson["result"].IsMap()) {
            passData.SetError("Invalid reply format");
            passData.SetApiResponse(reply.Content());
            return passData;
        }
        bool isValid;
        if (!NJson::ParseField(replyJson["result"]["active"], isValid, true)) {
            isValid = true;
        }
        TString dateEnd;
        if (!NJson::ParseField(replyJson["result"]["data"]["DATE_END"], dateEnd, true)) {
            passData.SetError("Incorrect reply format");
            return passData;
        }
        TInstant until;
        TString error;
        if (!TryParseLocalDateTime(dateEnd, until, error) && !TInstant::TryParseIso8601(dateEnd, until)) {
            passData.SetError("Could not parse date. " + error);
            return passData;
        }
        passData.SetValidUntil(until);
        if (until < minUntil) {
            isValid = false;
        }
        passData.SetIsValid(isValid);
        return passData;
    });
}

NThreading::TFuture<TCovidPassClient::TPassData> TCovidPassMoscowRegionClient::GetPassData(const TString& pass, const TInstant minUntil) const {
    TString error;
    if (!CheckAndUpdateToken(error)) {
        TCovidPassClient::TPassData passData;
        passData.SetError(error);
        passData.SetIsValid(false);
        return NThreading::MakeFuture(passData);
    }

    TString upperPass = ToUpperUTF8(pass);
    SubstGlobal(upperPass, "-", "");
    const TString normalizedPass = TokenizeString(0, upperPass.length() - 8, 4, '-', upperPass);

    NNeh::THttpRequest request;
    request
        .SetUri(Config.GetApiPath() + normalizedPass)
        .SetRequestType("GET")
        .AddHeader("Authorization", Token);

    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([normalizedPass, minUntil](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        TCovidPassClient::TPassData passData;
        passData.SetPassToken(normalizedPass);
        NJson::TJsonValue replyJson;
        if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            passData.SetError("Bad response, code " + ToString(reply.Code()) + ". Error: " + reply.ErrorMessage());
            passData.SetApiResponse(replyJson);
            passData.SetIsValid(false);
            return passData;
        }
        ParseJsonReply(replyJson, minUntil, passData);
        return passData;
    });
}

NThreading::TFuture<TCovidPassClient::TPassData> TCovidPassGosuslugiClient::GetPassData(const TString& pass, const TInstant minUntil) const {
    TString normalizedPass;
    if (pass.length() < 30) {
        TString upperPass = ToUpperUTF8(pass);
        SubstGlobal(upperPass, "-", "");
        normalizedPass = TokenizeString(0, upperPass.length(), 4, '-', upperPass);
    } else {
        TString lowerPass = ToLowerUTF8(pass);
        SubstGlobal(lowerPass, "-", "");
        normalizedPass = TokenizeString(4, lowerPass.length() - 8, 4, '-', lowerPass);
    }
    NNeh::THttpRequest request;
    request
        .SetUri(Config.GetApiPath()+normalizedPass)
        .SetRequestType("GET")
        .AddHeader("User-Agent", "mos.carsharing");
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([normalizedPass, minUntil](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        TCovidPassClient::TPassData passData;
        passData.SetPassToken(normalizedPass);
        NJson::TJsonValue replyJson;
        if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            passData.SetApiResponse(replyJson);
            passData.SetError("Bad response, code " + ToString(reply.Code()) + ". Error: " + reply.ErrorMessage());
            passData.SetIsValid(false);
            return passData;
        }
        ParseJsonReply(replyJson, minUntil, passData);
        return passData;
    });
}

template <>
NJson::TJsonValue NJson::ToJson(const TCovidPassClient::TPassData& object) {
    NJson::TJsonValue result;
    result["pass_token"] = object.GetPassToken();
    result["car_id"] = object.GetCarId();
    result["is_valid"] = object.GetIsValid();
    result["valid_until"] = NJson::ToJson(object.GetValidUntil());
    result["api_response"] = object.GetApiResponse();
    result["error"] = NJson::ToJson(NJson::Nullable(object.GetError()));
    return result;
}
