#include "client.h"

#include <rtline/library/json/parse.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/str.h>

Y_UNIT_TEST_SUITE(CovidPassClient) {
    Y_UNIT_TEST(CovidPassGosuslugiType1) {
        TStringStream ss;
        ss << "{\"minutes\":0,\"created\":\"13.05.2020 09:06:43\",\"hours\":0,\"type\":1,\"parameters\":[{\"attributes\":[],";
        ss << "\"key\":\"Действителен с\",\"type\":\"String\",\"value\":\"13.05.2020\",\"order\":1},\n{\"attributes\":[],\"key\":\"по\",\"type\":\"String\",";
        ss << "\"value\":\"31.05.2020\",\"order\":2},{\"attributes\":[],\"key\":\"Номер и Серия документа\",\"type\":\"String\",\"value\":\"XXXXXXXXXX\",\"order\":3},";
        ss << "\n{\"attributes\":[],\"key\":\"Причина нахождения вне дома\",\"type\":\"String\",\"value\":\"Цифровой пропуск для работающих\",\"order\":4},";
        ss << "\n{\"attributes\":[],\"key\":\"Номер ТС\",\"type\":\"String\",\"value\":\"\",\"order\":5},{\"attributes\":[],\"key\":\"Наименование организации\",";
        ss << "\"type\":\"String\",\"value\":\"ОБЩЕСТВО С ОГРАНИЧЕННОЙ ОТВЕТСТВЕННОСТЬЮ \\\"МАЙНДБОКС\\\"\",\"order\":6},{\"attributes\":[],";
        ss << "\n\"key\":\"Наименование региона\",\"type\":\"String\",\"value\":\"Москва\",\"order\":8}],\n\"birthDate\":\"\",\"valid\":1,";
        ss << "\"lastName\":\"Л.\",\"status\":\"действителен\",\"firstName\":\"\",\"middleName\":\"\"}" << Endl;

        NJson::TJsonValue replyJson;
        NJson::ReadJsonFastTree(ss.Str(), &replyJson);

        TCovidPassClient::TPassData passData;
        ParseJsonReply(replyJson, TInstant::Seconds(1589760000), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), true);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1590883200);

        ParseJsonReply(replyJson, TInstant::Seconds(1596240000), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), false);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1590883200);
    }

    Y_UNIT_TEST(CovidPassGosuslugiType2) {
        TStringStream ss;
        ss << "{\"minutes\":9,\"created\":\"20.04.2020 00:00:00\",\"hours\":386,\"type\":2,\"parameters\":[{\"attributes\":[],";
        ss << "\"key\":\"Действителен с\",\"type\":\"Date\",\"value\":\"2020-04-20\",\"order\":1},{\"attributes\":[],\"key\":\"по\",";
        ss << "\"type\":\"Date\",\"value\":\"2020-06-04\",\"order\":2},{\"attributes\":[],\"key\":\"Номер документа\",\"type\":\"String\",\"value\":\"***41\",\"order\":4},";
        ss << " {\"attributes\":[],\"key\":\"Серия документа\",\"type\":\"String\",\"value\":\"46**\",\"order\":3},{\"attributes\":[],";
        ss << " \"key\":\"Вид деятельности\",\"type\":\"String\",\"value\":\"Непрерывно действующие организации\",\"order\":6},{\"attributes\":[],";
        ss << " \"key\":\"Наименование организации\",\"type\":\"String\",\"value\":\"ФИНАНСОВОЕ УПРАВЛЕНИЕ АДМИНИСТРАЦИИ ГОРОДСКОГО ОКРУГА МЫТИЩИ\",\"order\":5},{\"attributes\":[],";
        ss << " \"key\":\"Должность\",\"type\":\"String\",\"value\":\"главный аналитик\",\"order\":7}],\"valid\":1,";
        ss << " \"lastName\":\"Г***\",\"status\":\"действителен\",\"firstName\":\"Елена\",\"middleName\":\"Владимировна\"}" << Endl;
        NJson::TJsonValue replyJson;
        NJson::ReadJsonFastTree(ss.Str(), &replyJson);

        TCovidPassClient::TPassData passData;
        ParseJsonReply(replyJson, TInstant::Seconds(1589760000), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), true);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1591228800);

        ParseJsonReply(replyJson, TInstant::Seconds(1596240000), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), false);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1591228800);
    }

    Y_UNIT_TEST(CovidPassInvalid) {
        TStringStream ss;
        ss << "{\"type\":1,\"valid\":0,\"status\":\"не действителен\",\"firstName\":\"Илья\",\"lastName\":\"В.\",\"middleName\":\"\",\"created\":\"19.04.2020 ";
        ss << "05:04:10\",\"hours\":0,\"minutes\":0,\"birthDate\":\"\",\"parameters\":[{\"key\":\"Действителен с\",\"value\":\"19.04.2020\",\"type\":\"String\",\"order\":1,\"attributes\":[]},";
        ss << "{\"key\":\"по\",\"value\":\"19.04.2020\",\"type\":\"String\",\"order\":2,\"attributes\":[]},\n   {\"key\":\"Номер и Серия ";
        ss << "документа\",\"value\":\"XXXXXXXXX\",\"type\":\"String\",\"order\":3,\"attributes\":[]},{\"key\":\"Причина нахождения вне дома\",\"value\":\"Разовая поездка в иных ";
        ss << "целях\",\"type\":\"String\",\"order\":4,\"attributes\":[]},\n   {\"key\":\"Наименование региона\",\"value\":\"Москва\",\"type\":\"String\",\"order\":8,\"attributes\":[]},";
        ss << "{\"key\":\"Карта «Тройка»\",\"value\":\"002XXXXX02\",\"type\":\"String\",\"order\":9,\"attributes\":[]}]}" << Endl;
        NJson::TJsonValue replyJson;
        NJson::ReadJsonFastTree(ss.Str(), &replyJson);

        TCovidPassClient::TPassData passData;
        ParseJsonReply(replyJson, TInstant::Seconds(0), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), false);

        TString incorrectFormat = "{ \"blah\" : \"blah blah\" }";
        NJson::ReadJsonFastTree(incorrectFormat, &replyJson);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), false);
    }

    Y_UNIT_TEST(CovidPassMosRegType) {
        TStringStream ss;
        ss << "{\"minutes\":0,\"created\":\"14.05.2020 17:59:49\",\"hours\":12,\"type\":1,\n    \"parameters\":[{\"key\":\"Код пропуска\",\"type\":\"String\",\"value\":\"200550Q2RKPG9GVT\",\"order\":1},";
        ss << "{\"key\":\"Контактные данные\",\"type\":\"String\",\"value\":\"9631111111\",\"order\":2},{\"key\":\"Удостоверяющий личность документ\",\"type\":\"String\",\"value\":\"11** ****11\",\"order\":3},";
        ss << "{\"key\":\"Причина нахождения вне дома\",\"type\":\"String\",\"value\":\"Иная причина: Магазин Тестовый\",\"order\":5},{\"key\":\"Цифровой пропуск ";
        ss << "создан\",\"type\":\"Time\",\"value\":\"17:59:00\",\"order\":6},{\"key\":\"Планируемое время нахождения вне дома\",\"type\":\"String\",\"value\":\"Длительная поездка (12:00)\",\"order\":7},";
        ss << "{\"key\":\"Наименование места назначения\",\"type\":\"String\",\"value\":\"Магазин Тестовый\",\"order\":13},{\"key\":\"Действует до\",\"type\":\"String\",\"value\":\"20.05.2020\",\"order\":14},";
        ss << "{\"key\":\"Оставшийся лимит по времени\",\"type\":\"String\",\"value\":\"02д 00ч 44м\",\"order\":15},{\"key\":\"Номер ТС\",\"type\":\"String\",\"value\":\"АА111АА750\",\"order\":12}],\"valid\":1,";
        ss << "\"birthDate\":\"06.08.2009\",\"lastName\":\"Пропуск\",\"status\":\"действителен\",\"firstName\":\"по смс\"}" << Endl;
        NJson::TJsonValue replyJson;
        NJson::ReadJsonFastTree(ss.Str(), &replyJson);

        TCovidPassClient::TPassData passData;
        ParseJsonReply(replyJson, TInstant::Seconds(1589846400), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), true);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1589932800);

        ParseJsonReply(replyJson, TInstant::Seconds(1599932800), passData);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetIsValid(), false);
        UNIT_ASSERT_VALUES_EQUAL(passData.GetValidUntil().Seconds(), 1589932800);
    }

    Y_UNIT_TEST(PassTokenizer) {
        TString str = "81e15372b0fc458f9eb6e04a4ef29c50";
        UNIT_ASSERT_VALUES_EQUAL(TokenizeString(0, 100, 4, '-', str), "81e1-5372-b0fc-458f-9eb6-e04a-4ef2-9c50");
        str = "12345";
        UNIT_ASSERT_VALUES_EQUAL(TokenizeString(0, str.length(), 1, '-', str), "1-2-3-4-5");
        str = "81e15372b0fc458f9eb6e04a4ef29c50";
        UNIT_ASSERT_VALUES_EQUAL(TokenizeString(4, str.length() - 8, 4, '-', str), "81e15372-b0fc-458f-9eb6-e04a4ef29c50");
        str = "7d86ac773dfc4556829ffbab8d6c5871";
        UNIT_ASSERT_VALUES_EQUAL(TokenizeString(4, str.length() - 8, 4, '-', str), "7d86ac77-3dfc-4556-829f-fbab8d6c5871");
        str = "3105PK80MCBCTHXX";
        UNIT_ASSERT_VALUES_EQUAL(TokenizeString(0, str.length(), 4, '-', str), "3105-PK80-MCBC-THXX");
    }
}
