#include "config.h"

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <util/folder/path.h>
#include <util/stream/file.h>


namespace {
    bool ParseToken(const TString& key, const TYandexConfig::Section* section, TString& token, TString& path) {
        token = section->GetDirectives().Value(key, token);
        path = section->GetDirectives().Value(key + "Path", path);
        if (!token && !path) {
            ERROR_LOG << "neither " << key << " nor " << key << " path are defined" << Endl;
            return false;
        }
        if (!token) {
            if (!TFsPath(path).Exists()) {
                ERROR_LOG << "Incorrect path for " << key << "'s apikey in '" << key << "Path'" << Endl;
                return false;
            }
            TFileInput fi(path);
            token = fi.ReadAll();
        }
        return true;
    }
}

namespace NDrive {

    bool TDadataConfig::Init(const TYandexConfig::Section* section) {
        Uri = section->GetDirectives().Value("Uri", Uri);
        UriPath = section->GetDirectives().Value("UriPath", UriPath);
        {
            const TYandexConfig::TSectionsMap children = section->GetAllChildren();
            auto it = children.find("RequestConfig");
            if (it != children.end()) {
                RequestConfig.InitFromSection(it->second);
            }
        }
        RequestTimeout  = section->GetDirectives().Value("RequestTimeout", RequestTimeout);
        return ParseToken("Token", section, Token, TokenPath)
            && ParseToken("Secret", section, Secret, SecretPath);
    }

    void TDadataConfig::ToString(IOutputStream& os) const {
        os << "Uri: " << Uri << Endl;
        os << "UriPath: " << UriPath << Endl;
        if (TokenPath) {
            os << "TokenPath: " << TokenPath << Endl;
        }
        if (SecretPath) {
            os << "SecretPath: " << SecretPath << Endl;
        }
        os << "<RequestConfig>" << Endl;
        RequestConfig.ToString(os);
        os << "</RequestConfig>" << Endl;
        os << "RequestTimeout : " << RequestTimeout << Endl;
    }

    void TDadataConfig::Authorize(NNeh::THttpRequest& request) const {
        request.AddHeader("Authorization", "Token " + Token);
        request.AddHeader("X-Secret", Secret);
    }

    TAtomicSharedPtr<TDadataConfig> TDadataConfig::ParseFromString(const TString& configStr) {
        TAnyYandexConfig config;
        if (!config.ParseMemory(configStr.data())) {
            ERROR_LOG << "Fail to parse config:" + configStr << Endl;
            return nullptr;
        }
        auto result = MakeAtomicShared<TDadataConfig>();
        if (!result->Init(config.GetRootSection())) {
            return nullptr;
        }
        return result;
    }

}
