#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_reader.h>

TDatasyncClient::TResponse::TResponse(TStringBuf content, ui16 code)
    : Code(code)
{
    if (!NJson::ReadJsonFastTree(content, &Value)) {
        Value["message"] = content;
    }
}

const TString& TDatasyncClient::TResponse::GetError() const {
    return Value["error"].GetString();
}

const TString& TDatasyncClient::TResponse::GetDescription() const {
    return Value["description"].GetString();
}

const TString& TDatasyncClient::TResponse::GetMessage() const {
    return Value["message"].GetString();
}

TDatasyncClient::TDatasyncClient(const TDatasyncClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm)
    : Config(config)
    , Tvm(tvm)
{
    if (!Tvm && config.GetTvmConfig().GetSource()) {
        NTvmAuth::NTvmApi::TClientSettings tvmSettings;
        tvmSettings.SetSelfTvmId(config.GetTvmConfig().GetSource());
        tvmSettings.EnableServiceTicketsFetchOptions(config.GetTvmConfig().GetToken(), NTvmAuth::NTvmApi::TClientSettings::TDstVector{ config.GetTvmConfig().GetDestination() });
        Tvm = MakeAtomicShared<NTvmAuth::TTvmClient>(tvmSettings, NTvmAuth::TDevNullLogger::IAmBrave());
    }
    AsyncDelivery = new TAsyncDelivery;
    AsyncDelivery->Start(config.GetRequestConfig().GetThreadsStatusChecker(), config.GetRequestConfig().GetThreadsSenders());
    Requester.Reset(new NNeh::THttpClient(AsyncDelivery));
    Requester->RegisterSource("datasync", TString(config.GetHost()), config.GetPort(), config.GetRequestConfig(), false);
}

TDatasyncClient::~TDatasyncClient() {
    AsyncDelivery->Stop();
}

NThreading::TFuture<TDatasyncClient::TResponse> TDatasyncClient::Get(const TString& collection, const TString& key, ui64 uid) const {
    if (!uid) {
        return NThreading::TExceptionFuture() << "uid is zero";
    }

    auto deadline = ConstructDeadline();
    auto request = ConstructBaseRequest(collection, key, uid);

    return Requester->SendAsync(request, deadline).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        return TResponse(reply.Content(), reply.Code());
    });
}

NThreading::TFuture<TDatasyncClient::TResponse> TDatasyncClient::Update(const TString& collection, const TString& key, ui64 uid, const NJson::TJsonValue& payload) const {
    if (!uid) {
        return NThreading::TExceptionFuture() << "uid is zero";
    }

    auto deadline = ConstructDeadline();
    auto request = ConstructBaseRequest(collection, key, uid);
    request.SetPostData(payload.GetStringRobust());
    request.SetRequestType("PUT");
    return Requester->SendAsync(request, deadline).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        return TResponse(reply.Content(), reply.Code());
    });
}

NThreading::TFuture<TDatasyncClient::TResponse> TDatasyncClient::Delete(const TString& collection, const TString& key, ui64 uid) const {
    if (!uid) {
        return NThreading::TExceptionFuture() << "uid is zero";
    }

    auto deadline = ConstructDeadline();
    auto request = ConstructBaseRequest(collection, key, uid);
    request.SetRequestType("DELETE");
    return Requester->SendAsync(request, deadline).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        return TResponse(reply.Content(), reply.Code());
    });
}

NNeh::THttpClient::TGuard TDatasyncClient::PerformCollectionRequest(const TString& collectionName, const TString& documentKey, const ui64 uid, THolder<NNeh::THttpAsyncReport::ICallback> && callback) const {
    auto request = ConstructBaseRequest(collectionName, documentKey, uid);
    return Requester->Send(request, ConstructDeadline(), std::move(callback));
}

NUtil::THttpReply TDatasyncClient::PerformSyncCollectionRequest(const TString& collectionName, const TString& documentKey, const ui64 uid) const {
    auto request = ConstructBaseRequest(collectionName, documentKey, uid);
    return Requester->SendMessageSync(request, ConstructDeadline());
}

NNeh::THttpClient::TGuard TDatasyncClient::PerformUpdateRequest(const TString& collectionName, const TString& documentKey, const ui64 uid, const NJson::TJsonValue& payload, THolder<NNeh::THttpAsyncReport::ICallback> && callback) const {
    auto request = ConstructBaseRequest(collectionName, documentKey, uid);
    request.SetPostData(payload.GetStringRobust());
    request.SetRequestType("PUT");
    return Requester->Send(request, ConstructDeadline(), std::move(callback));
}

NUtil::THttpReply TDatasyncClient::PerformSyncUpdateRequest(const TString& collectionName, const TString& documentKey, const ui64 uid, const NJson::TJsonValue& payload) const {
    auto request = ConstructBaseRequest(collectionName, documentKey, uid);
    request.SetPostData(payload.GetStringRobust());
    request.SetRequestType("PUT");
    return Requester->SendMessageSync(request, ConstructDeadline());
}

NNeh::THttpRequest TDatasyncClient::ConstructBaseRequest(const TString& collectionName, const TString& documentKey, const ui64 uid) const {
    Y_ENSURE(uid);
    // Get endpoint path
    TString endpointRoute = Config.GetBaseRoute();
    if (!endpointRoute.EndsWith("/")) {
        endpointRoute += "/";
    }
    endpointRoute += collectionName + "/" + documentKey;
    ui32 destinationClientId = std::max(Config.GetDestinationClientId(), Config.GetTvmConfig().GetDestination());

    // Construct request object
    NNeh::THttpRequest request;
    request.SetUri(endpointRoute);
    request.AddHeader("X-Uid", ToString(uid));
    if (Tvm) {
        request.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(destinationClientId));
    }
    return request;
}

TInstant TDatasyncClient::ConstructDeadline(TDuration timeout /*= TDuration::Zero()*/) const {
    return Now() + (timeout ? timeout : Config.GetRequestConfig().GetGlobalTimeout());
}
