#include "client.h"

bool TDatasyncQueueEntry::DeserializeWithDecoder(const TDatasyncEntryDecoder& decoder, const TConstArrayRef<TStringBuf> & values, const IHistoryContext*) {
    READ_DECODER_VALUE(decoder, values, AssignmentId);
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, CreatedAt);
    READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ProcessAt);
    READ_DECODER_VALUE_DEF(decoder, values, Attempt, 1);
    return true;
}

bool TDatasyncQueueEntry::Parse(const NStorage::TTableRecord& row) {
    TDatasyncEntryDecoder decoder;
    return decoder.DeserializeFromTableRecord(*this, row);
}

NStorage::TTableRecord TDatasyncQueueEntry::SerializeToTableRecord() const {
    NStorage::TTableRecord record;

    record.Set("assignment_id", AssignmentId);
    record.Set("attempt", Attempt);

    if (CreatedAt != TInstant::Zero()) {
        record.Set("created_at", CreatedAt.ToString());
    }
    if (ProcessAt != TInstant::Zero()) {
        record.Set("process_at", ProcessAt.ToString());
    }

    return record;
}

NDrive::TEntitySession TDatasyncQueueClient::BuildSession(bool readonly /* = false */) const {
    return NDrive::TEntitySession(Database->CreateTransaction(readonly));
}

TMap<TString, TDatasyncQueueEntry> TDatasyncQueueClient::Get(ui64 amount, NDrive::TEntitySession& session) const {
    TQueryOptions queryOptions;
    queryOptions.SetOrderBy({"process_at"}).SetDescending(false);
    queryOptions.SetLimit(amount);
    return FetchInfo(session, queryOptions).GetResult();
}

bool TDatasyncQueueClient::Delete(const TString& assignmentId, NDrive::TEntitySession& session) const {
    return Remove(assignmentId, session);
}

bool TDatasyncQueueClient::Push(const TString& assignmentId, NDrive::TEntitySession& session) const {
    TInstant now = TInstant::Now();

    TDatasyncQueueEntry assignment;
    assignment.SetAssignmentId(assignmentId);
    assignment.SetCreatedAt(now);
    assignment.SetProcessAt(now);
    assignment.SetAttempt(1);

    return Insert(assignment, session);
}

bool TDatasyncQueueClient::Return(TDatasyncQueueEntry assignment, NDrive::TEntitySession& session) const {
    assignment.SetProcessAt(TInstant::Now());
    assignment.SetAttempt(assignment.GetAttempt() + 1);

    return Upsert(assignment, session);
}

ui64 TDatasyncQueueClient::Size(NDrive::TEntitySession& session) const {
    auto transaction = session.GetTransaction();
    ui64 size;
    TRecordsSet records;
    TString query = "SELECT COUNT(1) FROM " + GetTableName();
    auto queryResult = Yensured(transaction)->Exec(query, &records);
    if (
        !queryResult ||
        !queryResult->IsSucceed() ||
        records.size() != 1 ||
        !records.GetRecords().front().Has("count") ||
        !TryFromString(records.GetRecords().front().Get("count"), size)
    ) {
        throw TCodedException(HTTP_INTERNAL_SERVER_ERROR)
            .AddInfo("query", query)
            .AddInfo("transaction", transaction->GetErrors().GetReport())
            .AddInfo("type", "TDatasyncQueueClient::Size");
    }
    return size;
}
