#include <drive/library/cpp/datasync_queue/client.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <kernel/daemon/config/daemon_config.h>
#include <rtline/library/storage/postgres/table_accessor.h>
#include <rtline/util/types/uuid.h>

#include <util/system/env.h>

bool EmptyQueue(const TDatasyncQueueClient& queueManager) {
    auto session = queueManager.BuildSession();
    while (auto assignment = queueManager.Get(1, session)) {
        if (!queueManager.Delete(assignment.begin()->first, session)) {
            return false;
        }
    }
    if (!session.Commit()) {
        return false;
    }
    return true;
}

NStorage::IDatabase::TPtr GetDatabase() {
    TString host = GetEnv("Host");
    UNIT_ASSERT_C(host, "env Host undefined");
    TString password = GetEnv("Password");
    UNIT_ASSERT_C(host, "env Password undefined");

    TStringBuilder builder;
    builder << "<Config>" << Endl;
    builder << "    Type: Postgres" << Endl;
    builder << "    ConnectionString:";
    builder << " host=" << host;
    builder << " port=6432";
    builder << " target_session_attrs=read-write";
    builder << " dbname=extmaps-carsharing-testing";
    builder << " user=carsharing";
    builder << " password=" << password << Endl;
    builder << "    UseBalancing: true" << Endl;
    builder << "</Config>";
    TAnyYandexConfig rawConfig;

    UNIT_ASSERT_C(rawConfig.ParseMemory(builder), "Failed to parse section string");

    TPostgresConfig config;
    config.Init(rawConfig.GetFirstChild("Config"));

    return config.ConstructDatabase();
}

Y_UNIT_TEST_SUITE(DatasyncQueueClient) {
    Y_UNIT_TEST(Essentials) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId = NUtil::CreateUUID();
        auto session = datasyncQueueClient.BuildSession();

        auto assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(!assignment);

        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId, session));
        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId));

        UNIT_ASSERT(datasyncQueueClient.Delete(assignmentId, session));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment.empty());
    }

    Y_UNIT_TEST(TimestampCorrectness) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId = NUtil::CreateUUID();
        auto session = datasyncQueueClient.BuildSession();

        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId, session));
        auto assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);

        TInstant stamp = Now();
        assignment[assignmentId].SetCreatedAt(stamp);

        UNIT_ASSERT(datasyncQueueClient.Return(assignment[assignmentId], session));
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment[assignmentId].GetCreatedAt() == stamp);
    }

    Y_UNIT_TEST(Order) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId1 = NUtil::CreateUUID();
        auto assignmentId2 = NUtil::CreateUUID();

        auto session = datasyncQueueClient.BuildSession();
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId1, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId2, session));

        auto assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId1));
        UNIT_ASSERT(datasyncQueueClient.Delete(assignmentId1, session));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId2));
        UNIT_ASSERT(datasyncQueueClient.Delete(assignmentId2, session));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment.empty());
    }

    Y_UNIT_TEST(Return) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId1 = NUtil::CreateUUID();
        auto assignmentId2 = NUtil::CreateUUID();

        auto session = datasyncQueueClient.BuildSession();
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId1, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId2, session));

        auto assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId1));
        UNIT_ASSERT(assignment[assignmentId1].GetAttempt() == 1);
        UNIT_ASSERT(assignment[assignmentId1].GetCreatedAt() == assignment[assignmentId1].GetProcessAt());
        UNIT_ASSERT(datasyncQueueClient.Return(assignment[assignmentId1], session));

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId2));
        datasyncQueueClient.Delete(assignmentId2, session);

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment);
        UNIT_ASSERT(assignment.contains(assignmentId1));
        UNIT_ASSERT(assignment[assignmentId1].GetAttempt() == 2);
        UNIT_ASSERT(assignment[assignmentId1].GetCreatedAt() != assignment[assignmentId1].GetProcessAt());
        datasyncQueueClient.Delete(assignmentId1, session);

        assignment = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignment.empty());
    }

    Y_UNIT_TEST(Batching) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId1 = NUtil::CreateUUID();
        auto assignmentId2 = NUtil::CreateUUID();
        auto assignmentId3 = NUtil::CreateUUID();

        auto session = datasyncQueueClient.BuildSession();
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId1, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId2, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId3, session));

        auto assignments = datasyncQueueClient.Get(2, session);
        UNIT_ASSERT(assignments.size() == 2);
        UNIT_ASSERT(assignments.contains(assignmentId1));
        UNIT_ASSERT(assignments.contains(assignmentId2));

        assignments = datasyncQueueClient.Get(3, session);
        UNIT_ASSERT(assignments.size() == 3);
        UNIT_ASSERT(assignments.contains(assignmentId1));
        UNIT_ASSERT(assignments.contains(assignmentId2));
        UNIT_ASSERT(assignments.contains(assignmentId3));
        datasyncQueueClient.Delete(assignmentId1, session);
        datasyncQueueClient.Delete(assignmentId2, session);

        assignments = datasyncQueueClient.Get(3, session);
        UNIT_ASSERT(assignments.size() == 1);
        UNIT_ASSERT(assignments.contains(assignmentId3));
        datasyncQueueClient.Delete(assignmentId3, session);

        assignments = datasyncQueueClient.Get(1, session);
        UNIT_ASSERT(assignments.empty());
    }

    Y_UNIT_TEST(Size) {
        auto datasyncQueueClient = TDatasyncQueueClient(GetDatabase());
        UNIT_ASSERT_C(EmptyQueue(datasyncQueueClient), "Failed to prepare the queue");

        auto assignmentId1 = NUtil::CreateUUID();
        auto assignmentId2 = NUtil::CreateUUID();
        auto assignmentId3 = NUtil::CreateUUID();

        auto session = datasyncQueueClient.BuildSession();
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId1, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId2, session));
        UNIT_ASSERT(datasyncQueueClient.Push(assignmentId3, session));

        auto size = datasyncQueueClient.Size(session);
        UNIT_ASSERT_VALUES_EQUAL(size, 3);
    }
}
