#include "signature.h"

#include <drive/library/cpp/openssl/rsa.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <openssl/err.h>
#include <openssl/evp.h>

TString NDrive::GetDeeplinkSignature(const TString& deeplink, NOpenssl::TEVP_PKEYPtr privateKey) {
    if (!privateKey) {
        Cerr << "null PrivateKey" << Endl;
        return {};
    }
    auto md = EVP_get_digestbyname("sha256");
    if (!md) {
        Cerr << "EVP_get_digestbyname sha256 failed: " << ERR_reason_error_string(ERR_get_error()) << Endl;
        return {};
    }
    auto in = NOpenssl::TBIOPtr(BIO_new_mem_buf(deeplink.data(), deeplink.size()));
    auto bmd = NOpenssl::TBIOPtr(BIO_new(BIO_f_md()));
    auto input = BIO_push(bmd.Get(), in.Get());
    if (!input) {
        Cerr << "BIO_push failed: " << ERR_reason_error_string(ERR_get_error()) << Endl;
        return {};
    }

    EVP_MD_CTX* mctx = nullptr;
    if (!BIO_get_md_ctx(input, &mctx)) {
        Cerr << "BIO_get_md_ctx failed: " << ERR_reason_error_string(ERR_get_error()) << Endl;
        return {};
    }
    EVP_PKEY_CTX* pctx = nullptr;
    if (!EVP_DigestSignInit(mctx, &pctx, md, nullptr, privateKey.Get())) {
        Cerr << "EVP_DigestSignInit failed: " << ERR_reason_error_string(ERR_get_error()) << Endl;
        return {};
    }

    constexpr size_t size = 8 * 1024;
    unsigned char buf[size];
    while (BIO_pending(input) || !BIO_eof(input)) {
        auto got = BIO_read(input, buf, size);
        if (got < 0) {
            Cerr << "BIO_read failure: " << ERR_reason_error_string(ERR_get_error()) << Endl;
            return {};
        }
        if (got == 0) {
            break;
        }
    }

    size_t len = size;
    if (!EVP_DigestSignFinal(mctx, buf, &len)) {
        Cerr << "EVP_DigestSignFinal failure: " << ERR_reason_error_string(ERR_get_error()) << Endl;
        return {};
    }

    TStringBuf raw(reinterpret_cast<char*>(buf), len);
    return Base64Encode(raw);
}
