#include "client.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/quote/quote.h>

class TDiskSyncCallback : public IDocumentStorage::IDocumentStorageFile::ICallback {
public:
    TDiskSyncCallback(TString& data, TRWMutex& lock, TMessagesCollector& errors)
        : Data(data)
        , Guard(lock)
        , Errors(errors)
    {
    }

    void ProcessSuccess(const TString& data) override {
        Data = data;
    }

    void ProcessError(const TString& error) override {
        Errors.AddMessage("disk", error);
    }

private:
    TString& Data;
    TReadGuard Guard;
    TMessagesCollector& Errors;
};

bool TYandexDiskClient::TBinaryFile::GetDataByLink(const TString& link, TString& data, TMessagesCollector& errors) const {
    TRWMutex lock;
    TMessagesCollector localErrors;
    TBinaryFile::GetDataByLink(link, new TDiskSyncCallback(data, lock, localErrors), Owner);
    {
        TWriteGuard g(lock);
    }
    errors.MergeMessages(localErrors);
    return !localErrors.HasMessages();
}

class TYandexDiskClient::TBinaryFile::TDiskCallbackWrapper : public NNeh::THttpAsyncReport::ICallback {
public:
    TDiskCallbackWrapper(IDocumentStorage::IDocumentStorageFile::ICallback::TPtr callback, const TYandexDiskClient& owner)
        : Callback(callback)
        , Owner(owner)
    {}

    void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
        CHECK_WITH_LOG(reports.size() == 1);
        const auto& reply = reports[0];
        if (reply.GetHttpCode() == HTTP_OK) {
            Callback->ProcessSuccess(reports[0].GetReportSafe());
            return;
        }
        if (reply.GetHttpCode() == HTTP_FOUND) {
            for (const auto& header : reply.GetHeaders()) {
                if (header.Name() == "Location") {
                    TYandexDiskClient::TBinaryFile::GetDataByLink(header.Value(), Callback, Owner);
                    return;
                }
            }
        }
        Callback->ProcessError(ToString(reply.GetHttpCode()));
    }

private:
    IDocumentStorage::IDocumentStorageFile::ICallback::TPtr Callback;
    const TYandexDiskClient& Owner;
};

void TYandexDiskClient::TBinaryFile::GetDataByLink(const TString& link, IDocumentStorageFile::ICallback::TPtr callback, const TYandexDiskClient& owner) {
    TStringBuf scheme, host;
    ui16 port = 443;
    if (!link || !TryGetSchemeHostAndPort(link, scheme, host, port)) {
        callback->ProcessError("incorrect link " + link);
        return;
    }
    TString hostStr(host.data(), host.size());
    TString sourceName = hostStr + ":" + ToString(port);
    owner.CheckAndRegisterSource(sourceName, hostStr, port);

    NNeh::THttpRequest req;
    TStringBuf uri = GetPathAndQuery(link);
    TString uriStr(uri.data(), uri.size());
    req.SetUri(uriStr);
    const TInstant deadline = Now() + owner.GetConfig().GetRequestTimeout();

    std::multimap<TString, NNeh::THttpRequest> requests;
    requests.emplace(sourceName, req);
    owner.GetDownloadImpl()->SendPtr(requests, deadline, new TDiskCallbackWrapper(callback, owner));
}

void TYandexDiskClient::CheckAndRegisterSource(const TString& name, const TString& host, ui16 port) const {
    TGuard g(DownloadAgentGuard);
    if (!DownloadImpl->HasSource(name)) {
        DownloadImpl->RegisterSource(name, host, port, DownloadImpl.GetCommonConfig().GetRequestConfig(), port == 443);
        Logger.RegisterSource();
    }
}

NNeh::THttpRequest TYandexDiskClient::CreateCommonRequest(const TString& method, const TString& postData) const {
    NNeh::THttpRequest req;
    if (method == "POST" || method == "PUT") {
        req.SetPostData(TBlob::FromString(postData));
    }
    req.SetRequestType(method);
    req.AddHeader("Authorization", "OAuth " + Config.GetToken());
    req.AddHeader("Content-Type", "application/json; charset=utf-8");
    return req;
}

class TJsonCallback : public NNeh::THttpAsyncReport::ICallback {
public:
    TJsonCallback(NJson::TJsonValue& result, HttpCodes& code, TString& errorMessage)
        : Code(code)
        , Result(result)
        , ErrorMessage(errorMessage)
    {}

    void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
        CHECK_WITH_LOG(reports.size() == 1);
        const auto& result = reports.front();
        DEBUG_LOG << "Code: " << result.GetHttpCode() << Endl;
        Code = static_cast<HttpCodes>(result.GetHttpCode());
        if (!NJson::ReadJsonFastTree(*result.GetReport(), &Result)) {
            Code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR;
            ErrorMessage = "Incorrect json format";
        } else if (Code != HttpCodes::HTTP_OK) {
            ErrorMessage = ::ToString(Code) + " " + Result.GetStringRobust();
        }
    }
private:
    HttpCodes& Code;
    NJson::TJsonValue& Result;
    TString& ErrorMessage;
};

bool TYandexDiskClient::SendRequest(EYandexDiskOperationType operationType, const NNeh::THttpRequest& request, NJson::TJsonValue& reply, TMessagesCollector& errors) const {
    Logger.ProcessStart(operationType);
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    HttpCodes code;
    TString errorMessage;
    auto g = Impl->SendPtr(request, deadline, new TJsonCallback(reply, code, errorMessage));
    g.Wait();
    Logger.ProcessReply(code);
    if (code == HttpCodes::HTTP_OK) {
        DEBUG_LOG << reply.GetStringRobust() << Endl;
        return true;
    }
    Logger.ProcessError(operationType, errorMessage, errors);
    return false;
}

bool TYandexDiskClient::GetDownloadPath(const TString& path, TString& downloadPath, TMessagesCollector& errors) const {
    NJson::TJsonValue result;
    if (SendRequest(EYandexDiskOperationType::GetDownloadPath, CreateDownloadRequest(path), result, errors)) {
        downloadPath = result["href"].GetStringRobust();
        return true;
    }
    return false;
}

NNeh::THttpRequest TYandexDiskClient::CreateDownloadRequest(const TString& path) const {
    return CreateCommonRequest().SetUri("v1/disk/resources?path=" + UrlEscapeRet(path));
}

bool TYandexDiskClient::AddItem(const TString& startPath, const NJson::TJsonValue& item, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const {
    if (!item.Has("path")) {
        Logger.ProcessError(EYandexDiskOperationType::GetFiles, "Incorrect answer (\"path\" field is absent)", errors);
        return false;
    }
    TString localStartPath = startPath;
    if (localStartPath.Contains("disk:/")) {
        localStartPath = localStartPath.substr(6);
    }
    TString filePath = item["path"].GetStringRobust();
    if (filePath.Contains("disk:/")) {
        filePath = filePath.substr(6);
    }
    TString fileName = filePath.substr(localStartPath.size());
    if (fileName.StartsWith("/")) {
        fileName = fileName.substr(1);
    }

    IDocumentStorageFile::TPtr fileObject = new TBinaryFile(fileName, item, *this);
    fileList.emplace_back(fileObject);
    return true;
}

bool TYandexDiskClient::GetFiles(TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const {
    NJson::TJsonValue result;
    if (SendRequest(EYandexDiskOperationType::GetFiles, CreateGetFilesRequest(), result, errors)) {
        for (const auto& file : result["items"].GetArray()) {
            if (!AddItem("", file, fileList, errors)) {
                return false;
            }
        }
        return true;
    }
    return false;
}

bool TYandexDiskClient::GetFiles(const TString& path, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const {
    NJson::TJsonValue result;
    if (SendRequest(EYandexDiskOperationType::GetFiles, CreateGetFilesRequest(path), result, errors)) {
        if (result.Has("_embedded")) {
            for (const auto& file : result["_embedded"]["items"].GetArray()) {
                if (!file.Has("type") || file["type"].GetString() != "file") {
                    Logger.ProcessError(EYandexDiskOperationType::GetFiles, R"(Incorrect answer ("type" != "file"))", errors);
                    return false;
                }
                if (!AddItem(path, file, fileList, errors)) {
                    return false;
                }
            }
            return true;
        } else {
            if (!result.Has("type") || result["type"].GetString() != "file") {
                Logger.ProcessError(EYandexDiskOperationType::GetFiles, R"(Incorrect answer ("type" != "file"))", errors);
                return false;
            }
            return AddItem(path, result, fileList, errors);
        }
    }
    return false;
}

bool TYandexDiskClient::GetFilesRecursive(const TString& startPath, const TString& path, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const {
    NJson::TJsonValue result;
    if (SendRequest(EYandexDiskOperationType::GetFiles, CreateGetFilesRequest(path), result, errors)) {
        for (const auto& file : result["_embedded"]["items"].GetArray()) {
            if (!file.Has("type")) {
                Logger.ProcessError(EYandexDiskOperationType::GetFiles, "Incorrect answer (undefined \"type\")", errors);
                return false;
            }
            if (file["type"].GetString() == "dir") {
                if (!GetFilesRecursive(startPath, path + "/" + file["name"].GetStringRobust(), fileList, errors)) {
                    return false;
                }
                continue;
            }
            if (file["type"].GetString() != "file") {
                continue;
            }
            if (!AddItem(startPath, file, fileList, errors)) {
                return false;
            }
        }
        return true;
    }
    return false;
}

NNeh::THttpRequest TYandexDiskClient::CreateGetFilesRequest() const {
    return CreateCommonRequest().SetUri("v1/disk/resources/files?fields=" + Config.GetFileFields());
}

NNeh::THttpRequest TYandexDiskClient::CreateGetFilesRequest(const TString& path) const {
    return CreateCommonRequest().SetUri("v1/disk/resources?path=" + UrlEscapeRet(path));
}
